<?php

/*
Plugin Name: Deactivate Plugins per Page (MU Plugin)
Description: This is a must-use plugin that is required in order for the Deactivate Plugins per Page plugin to work. Must-use plugins run before normal plugins, and this allows us to deactivate normal plugins before they are loaded.
Author: Nikolay Nikolov
Author URI: https://nikolaydev.com/
Plugin URI: https://codecanyon.net/item/deactivate-plugins-per-page-improve-wordpress-performance/23801359
Version: 1.17.0
*/

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Gets an option value by name. If it does not exist, and we have a default value for it, returns the default value.
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string $name
 * @return mixed
 */
if ( ! function_exists( 'dppp_get_option' ) ) {
    function dppp_get_option( $name ) {
        $value = get_option( $name );
        if ( is_bool( $value ) && false === $value ) {
            $default_options = Array(
                'dppp-database-table-created' => 'no',
                'dppp-hide-group-plugins' => 'off',
                'dppp-debug-mode' => 'off',
                'dppp-inactive-plugins' => 'off',
                'dppp-completely-remove' => 'off',
                'dppp-groups-first' => 'off',
                'dppp-trailing-slash' => 'original',
                'dppp-affect-back-end' => 'off',
                'dppp-uri-parameters' => 'ignore-chosen',
                'dppp-uri-parameters-custom' => 'ignore-chosen',
                'dppp-chosen-uri-parameters' => 'fbclid, fb_action_ids, fb_action_types, fb_source, gclid, _ga, age-verified, ao_noptimize, usqp, '
                    . 'cn-reloaded, utm_medium, utm_expid, utm_term, utm_source, utm_campaign, utm_content, msclkid, dclid, ignorenitro, v',
                'dppp-section-show-hide-settings' => 'show',
                'dppp-section-show-hide-active-plugins' => 'show',
                'dppp-section-show-hide-inactive-plugins' => 'show',
                'dppp-section-show-hide-plugin-groups' => 'show',
                'dppp-plugin-icons' => 'on',
                'dppp-local-mode' => 'off',
                'dppp-rules-first' => 'off',
                'dppp-hide-plugins-in-group' => 'off',
                'dppp-priority' => 10,
                'dppp-affect-visual-editors' => 'off',
                'dppp-edit-post-parameter' => 'off',
                'dppp-admin-menu-cache-time' => 1800,
            );
            if ( array_key_exists( $name, $default_options ) ) {
                return $default_options[ $name ];
            } else {
                return false;
            }
        } else {
            return $value;
        }
    }
}

// We can skip filtering the option with this, so it does not cause an endless loop
$GLOBALS['dppp_skip_filter'] = 'no';

/*
 * These two variables will hold the $active_plugins array for the two options we are filtering, so we only do the calculations once
 * and not every time the options is called somewhere in the code.
 */
$GLOBALS['dppp_cache_option'] = 'dppp-not-set';
$GLOBALS['dppp_cache_site_option'] = 'dppp-not-set';

// This array will hold the plugins that this plugin has deactivated, so we can mark them in the debug box
$GLOBALS['dppp_deactivated_plugins'] = Array();

// The priority of the hooks to filter the active_plugins option. Sometimes changing this can solve rare plugin conflicts.
$GLOBALS['dppp_priority'] = dppp_get_option( 'dppp-priority' );

// Filters the options for the sidewide active plugins on a multisite and deactivates some plugins depending on the added rules
add_filter( 'site_option_active_sitewide_plugins', 'dppp_filter_active_plugins', $GLOBALS['dppp_priority'] );

// Filters the options for the active plugins and deactivates some plugins depending on the added rules
add_filter( 'option_active_plugins', 'dppp_filter_active_plugins', $GLOBALS['dppp_priority'] );

// Fixes an issue with some plugins becoming globally deactivated when other plugins change the actuve_plugins option
add_filter( 'pre_update_option_active_plugins', 'dppp_add_removed_active_plugins', 10, 3 );

/**
 * When the option active_plugins is updated, if we had deactivated some plugins (removed them from the option), we now add them again.
 * If we do not do that, when other plugins try to change the option, the plugins we had deactivated on the current page, will become globally deactivated.
 * @param mixed $value
 * @param string $option
 * @param mixed $old_value
 * @return mixed
 */
function dppp_add_removed_active_plugins( $value, $option, $old_value ) {
    if ( array_key_exists( 'dppp_deactivated_plugins', $GLOBALS ) && ! empty( $GLOBALS['dppp_deactivated_plugins'] ) ) {
        foreach ( $GLOBALS['dppp_deactivated_plugins'] as $plugin_file ) {
            if ( is_array( $value ) ) {
                $value[] = $plugin_file;
            } else {
                $value = Array( $plugin_file );
            }
        }
    }
    return $value;
}

/**
 * Filters the options for the active plugins and sidewide active plugins (on a multisite) and deactivates some plugins depending on the added rules.
 * @param mixed $active_plugins
 * @return mixed
 */
function dppp_filter_active_plugins( $active_plugins ) {

    // The deactivation rules will only affect GET requests on the front end that are not ajax requests (and will not affect the customizer)
    if ( 'no' === $GLOBALS['dppp_skip_filter'] && isset( $_SERVER['REQUEST_URI'] ) && ! ( defined( 'DOING_AJAX' ) && DOING_AJAX ) && ! dppp_is_rest_request()
        && isset( $_SERVER['REQUEST_METHOD'] ) && $_SERVER['REQUEST_METHOD'] === 'GET' && ! isset( $_GET['customize_changeset_uuid'] )
        && ( ! isset( $_GET['page'] ) || 'deactivate-site-plugins-per-page' !== $_GET['page'] ) && ! isset( $_GET['dppp-ignore'] ) ) {

        // Which of the options we are filtering, the one for the active plugins or the one for sitewide active plugins on a multisite
        $current_filter = current_filter();

        // If we have the value for $active_plugins cached for the current request already, we just return it, no need to calculate again
        if ( 'option_active_plugins' === $current_filter ) {
            if ( isset( $GLOBALS['dppp_cache_option'] )
                && ( ! is_string( $GLOBALS['dppp_cache_option'] ) || 'dppp-not-set' !== $GLOBALS['dppp_cache_option'] ) ) {
                return $GLOBALS['dppp_cache_option'];
            }
        } else {
            if ( isset( $GLOBALS['dppp_cache_site_option'] )
                && ( ! is_string( $GLOBALS['dppp_cache_site_option'] ) || 'dppp-not-set' !== $GLOBALS['dppp_cache_site_option'] ) ) {
                return $GLOBALS['dppp_cache_site_option'];
            }
        }

        // If the option to affect visual editors is disabled and we have detected one, we do no changes and stop everything
        if ( dppp_get_option( 'dppp-affect-visual-editors' ) === 'off'
            && ( isset( $_GET['elementor-preview'] ) || isset( $_GET['et_fb'] ) || isset( $_GET['vc_editable'] ) ) ) {

            // We cache the $active_plugins value for this request before returning it
            if ( 'option_active_plugins' === $current_filter ) {
                $GLOBALS['dppp_cache_option'] = $active_plugins;
            } else {
                $GLOBALS['dppp_cache_site_option'] = $active_plugins;
            }

            return $active_plugins;
        }

        // If local mode is enabled and this is not a local device, we do no changes and stop everything
        if ( dppp_get_option( 'dppp-local-mode' ) === 'on' && ! dppp_is_local_device() ) {

            // We cache the $active_plugins value for this request before returning it
            if ( 'option_active_plugins' === $current_filter ) {
                $GLOBALS['dppp_cache_option'] = $active_plugins;
            } else {
                $GLOBALS['dppp_cache_site_option'] = $active_plugins;
            }

            return $active_plugins;
        }

        $is_back_end = 'no';

        /*
         * On the back-end we do not deactivate plugins if the setting is not on, or if it is a page related to plugin control or update,
         * but still deactivate plugins on this plugin settings page if we are recreating the admin menu cache.
         */
        if ( is_admin() ) {
            if ( dppp_skip_this_admin_page()
                && strpos( $_SERVER['REQUEST_URI'], 'plugins.php?page=deactivate-plugins-per-page&dppp-recreate-cache=yes' ) === false ) {

                // We cache the $active_plugins value for this request before returning it
                if ( 'option_active_plugins' === $current_filter ) {
                    $GLOBALS['dppp_cache_option'] = $active_plugins;
                } else {
                    $GLOBALS['dppp_cache_site_option'] = $active_plugins;
                }

                return $active_plugins;
            }
            $is_back_end = 'yes';
        }

        /*
         * If it is a multisite we need to check if our plugin is activated either sitewide (for the network) or for the current site.
         * This must use plugin file is global for the whole network and it will remain active for all sites even if only one site has activated out plugin.
         * If our plugin is not active we return the normal option for active plugins without changes.
         */
        if ( is_multisite() ) {
            if ( 'option_active_plugins' === $current_filter ) {
                $site_active_plugins = $active_plugins;
                $previous_dppp_skip_filter = $GLOBALS['dppp_skip_filter'];
                $GLOBALS['dppp_skip_filter'] = 'yes';
                $network_active_plugins = get_site_option( 'active_sitewide_plugins' );
                $GLOBALS['dppp_skip_filter'] = $previous_dppp_skip_filter;
            } else {
                $network_active_plugins = $active_plugins;
                $previous_dppp_skip_filter = $GLOBALS['dppp_skip_filter'];
                $GLOBALS['dppp_skip_filter'] = 'yes';
                $site_active_plugins = get_option( 'active_plugins' );
                $GLOBALS['dppp_skip_filter'] = $previous_dppp_skip_filter;
            }
            if ( is_array( $site_active_plugins ) ) {
                $index = array_search( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php', $site_active_plugins );
                if ( false === $index ) {
                    $site_activated = 'no';
                } else {
                    $site_activated = 'yes';
                }
            } else {
                $site_activated = 'no';
            }
            if ( array_key_exists( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php', $network_active_plugins ) ) {
                $network_activated = 'yes';
            } else {
                $network_activated = 'no';
            }
            if ( $network_activated !== 'yes' && $site_activated !== 'yes' ) {

                // We cache the $active_plugins value for this request before returning it
                if ( 'option_active_plugins' === $current_filter ) {
                    $GLOBALS['dppp_cache_option'] = $active_plugins;
                } else {
                    $GLOBALS['dppp_cache_site_option'] = $active_plugins;
                }

                return $active_plugins;
            }

            // On the network dashboard we do not deactivate plugins if the setting is not on, or if it is a page related to plugin control or update
            if ( is_network_admin() ) {
                if ( dppp_skip_this_network_admin_page() ) {

                    // We cache the $active_plugins value for this request before returning it
                    if ( 'option_active_plugins' === $current_filter ) {
                        $GLOBALS['dppp_cache_option'] = $active_plugins;
                    } else {
                        $GLOBALS['dppp_cache_site_option'] = $active_plugins;
                    }

                    return $active_plugins;
                }
                $is_back_end = 'yes';
            }

            /*
             * On a multisite it is possible for the plugin to be active but the table to not be created (when network activated).
             * So we are creating this option when we create the table and now we check it.
             * If the table was not created we will do no changes and return. This will avoid an error
             * in the log for non-existing database table when we try to select from it below. And we are not using "show tables like"
             * to check for the table because this get_option method should be faster.
             */
            if ( get_option( 'dppp-database-table-created' ) !== 'yes' ) {

                // We cache the $active_plugins value for this request before returning it
                if ( 'option_active_plugins' === $current_filter ) {
                    $GLOBALS['dppp_cache_option'] = $active_plugins;
                } else {
                    $GLOBALS['dppp_cache_site_option'] = $active_plugins;
                }

                return $active_plugins;
            }

        // If this is not a multisite, we just need to check if our plugin is active for the current site
        } else {

            // If our plugin is not active we will do no changes even if this mu plugin remains
            if ( is_array( $active_plugins ) ) {
                $index = array_search( 'deactivate-plugins-per-page/deactivate-plugins-per-page.php', $active_plugins );
                if ( false === $index ) {

                    // We cache the $active_plugins value for this request before returning it
                    if ( 'option_active_plugins' === $current_filter ) {
                        $GLOBALS['dppp_cache_option'] = $active_plugins;
                    } else {
                        $GLOBALS['dppp_cache_site_option'] = $active_plugins;
                    }

                    return $active_plugins;
                }

            // If the option for the active plugins is not an array we will do no changes
            } else {

                // We cache the $active_plugins value for this request before returning it
                if ( 'option_active_plugins' === $current_filter ) {
                    $GLOBALS['dppp_cache_option'] = $active_plugins;
                } else {
                    $GLOBALS['dppp_cache_site_option'] = $active_plugins;
                }

                return $active_plugins;
            }
        }

        $is_mobile = 'unset';

        $use_custom_rules = 'yes';

        /*
         * Don't apply rules with custom URI selection on the back-end if either the affect back-end setting is off, or if we have the parameter
         * to ignore these rules in the URL, or if we are recreating the admin menu cache after turning on the admin menu cache setting.
         */
        if ( 'yes' === $is_back_end ) {
            if ( dppp_get_option( 'dppp-affect-back-end' ) === 'off'
                || strpos( $_SERVER['REQUEST_URI'], 'dppp-ignore-custom-rules=yes' ) !== false
                || strpos( $_SERVER['REQUEST_URI'], 'plugins.php?page=deactivate-plugins-per-page&dppp-recreate-cache=yes' ) !== false ) {
                $use_custom_rules = 'no';
            }
        }

        $decoded_uri = rawurldecode( $_SERVER['REQUEST_URI'] );

        // Based no the settings we may ignore all URI parameters when applying rules with Page URI selection
        $uri_for_page_rules = dppp_get_uri_for_page_rules( $decoded_uri );

        // Based no the settings we may ignore all or some URI parameters when applying rules with Custom URI selection
        $uri_for_custom_rules = dppp_get_uri_for_custom_rules( $decoded_uri );

        global $wpdb;

        /*
         * We get all the deactivation rules that are started.
         * We select all columns because sometimes we add new columns and this will avoid an error if it is not created yet.
         */
        $results_multi_array = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}dppp_plugin_deactivation_rules WHERE status = 'started'", ARRAY_A );

        // Caches the results from calling custom functions so if multiple rules use it, we call the function only once
        $custom_function_cache = Array();

        // If there are rules, for each rule we decide which plugins to deactivate (removing them from the active plugins array) on the current page
        if ( ! empty( $results_multi_array ) ) {
            foreach ( $results_multi_array as $results ) {
                if ( 'all' !== $results['devices'] ) {
                    if ( 'unset' === $is_mobile ){
                        $is_mobile = dppp_is_mobile();
                    }
                    if ( ( false === $is_mobile && 'mobile' === $results['devices'] ) || ( true === $is_mobile && 'desktop' === $results['devices'] ) ) {
                        continue;
                    }
                }

                // This logic deactivates plugins affected by global rules (deactivation on the whole front-end or back-end)
                if ( 'global' === $results['uri_type'] ) {
                    if ( ( '[dppp-front-end]' === $results['uri_value'] && 'no' === $is_back_end )
                        || ( '[dppp-back-end]' === $results['uri_value'] && 'yes' === $is_back_end ) ) {

                        // For plugin group rules, we put the plugins in an array and go through them. If it is a single plugin, it works too.
                        $plugin_files = explode( ';;', $results['plugin_file'] );
                        foreach ( $plugin_files as $plugin_file ) {

                            // Currently filtering the option for the sidewide active plugins on a multisite
                            if ( 'site_option_active_sitewide_plugins' === $current_filter ) {
                                if ( array_key_exists( $plugin_file, $active_plugins ) ) {
                                    unset( $active_plugins[ $plugin_file ] );
                                    if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                        $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                    }
                                }

                            // Currently filtering the option for active plugins on a single site
                            } else {
                                $key = array_search( $plugin_file, $active_plugins );
                                if ( false !== $key ) {
                                    unset( $active_plugins[ $key ] );
                                    if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                        $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                    }
                                }
                            }

                        }

                    }
                }

                // This logic deactivates plugins affected by page type rules (does not affect back-end regardless of settings)
                if ( 'page' === $results['uri_type'] && 'no' === $is_back_end ) {
                    $uri_values = explode( '[dppp-or]', $results['uri_value'] );
                    if ( 'selected' === $results['deactivation_type'] ) {
                        foreach ( $uri_values as $uri_value ) {
                            if ( $uri_for_page_rules === $uri_value && 'yes' === dppp_custom_function_allows( $results, $custom_function_cache ) ) {

                                // For plugin group rules, we put the plugins in an array and go through them. If it is a single plugin, it works too.
                                $plugin_files = explode( ';;', $results['plugin_file'] );
                                foreach ( $plugin_files as $plugin_file ) {

                                    // Currently filtering the option for the sidewide active plugins on a multisite
                                    if ( 'site_option_active_sitewide_plugins' === $current_filter ) {
                                        if ( array_key_exists( $plugin_file, $active_plugins ) ) {
                                            unset( $active_plugins[ $plugin_file ] );
                                            if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                                $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                            }
                                        }

                                    // Currently filtering the option for active plugins on a single site
                                    } else {
                                        $key = array_search( $plugin_file, $active_plugins );
                                        if ( false !== $key ) {
                                            unset( $active_plugins[ $key ] );
                                            if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                                $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                            }
                                        }
                                    }

                                }

                            }
                        }
                    } elseif ( 'except' === $results['deactivation_type'] ) {
                        if ( ! in_array( $uri_for_page_rules, $uri_values ) && 'yes' === dppp_custom_function_allows( $results, $custom_function_cache ) ) {

                            // For plugin group rules, we put the plugins in an array and go through them. If it is a single plugin, it works too.
                            $plugin_files = explode( ';;', $results['plugin_file'] );
                            foreach ( $plugin_files as $plugin_file ) {

                                // Currently filtering the option for the sidewide active plugins on a multisite
                                if ( 'site_option_active_sitewide_plugins' === $current_filter ) {
                                    if ( array_key_exists( $plugin_file, $active_plugins ) ) {
                                        unset( $active_plugins[ $plugin_file ] );
                                        if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                            $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                        }
                                    }

                                // Currently filtering the option for active plugins on a single site
                                } else {
                                    $key = array_search( $plugin_file, $active_plugins );
                                    if ( false !== $key ) {
                                        unset( $active_plugins[ $key ] );
                                        if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                            $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }

                // This logic deactivates plugins affected by custom type rules
                if ( 'custom' === $results['uri_type'] && 'yes' === $use_custom_rules ) {
                    $conditions = explode( '[dppp-or]', $results['uri_condition'] );
                    $values = explode( '[dppp-or]', $results['uri_value'] );
                    $current_uri_selected = 'no';

                    // We check all the conditions to see if the current uri is selected
                    for ( $loop = 0; $loop < count( $conditions ); $loop++ ) {

                        if ( ( 'equals' === $conditions[ $loop ] && $uri_for_custom_rules === $values[ $loop ] )
                            || ( 'does-not-equal' === $conditions[ $loop ] && $uri_for_custom_rules !== $values[ $loop ] )
                            || ( 'starts-with' === $conditions[ $loop ]
                                && substr( $uri_for_custom_rules, 0, strlen( $values[ $loop ] ) ) === $values[ $loop ] )
                            || ( 'does-not-start-with' === $conditions[ $loop ]
                                && substr( $uri_for_custom_rules, 0, strlen( $values[ $loop ] ) ) !== $values[ $loop ] )
                            || ( 'ends-with' === $conditions[ $loop ]
                                && substr( $uri_for_custom_rules, - ( strlen( $values[ $loop ] ) ) ) === $values[ $loop ] )
                            || ( 'does-not-end-with' === $conditions[ $loop ]
                                && substr( $uri_for_custom_rules, - ( strlen( $values[ $loop ] ) ) ) !== $values[ $loop ] )
                            || ( 'regular-expression' === $conditions[ $loop ] && preg_match( $values[ $loop ], $uri_for_custom_rules ) ) ) {
                            $current_uri_selected = 'yes';
                            break;
                        }

                        if ( 'contains' === $conditions[ $loop ] || 'does-not-contain' === $conditions[ $loop ] ) {

                            // If there are multiple values for this condition they all have to be contained (or not contained)
                            if ( strpos( $values[ $loop ], '--dppp-and-value--' ) !== false ) {

                                $condition_values = explode( '--dppp-and-value--', $values[ $loop ] );

                                if ( 'contains' === $conditions[ $loop ] ) {
                                    $all_contained = 'yes';
                                    foreach ( $condition_values as $condition_value ) {
                                        if ( strpos( $uri_for_custom_rules, $condition_value ) === false ) {
                                            $all_contained = 'no';
                                            break;
                                        }
                                    }
                                    if ( 'yes' === $all_contained ) {
                                        $current_uri_selected = 'yes';
                                        break;
                                    }
                                } elseif ( 'does-not-contain' === $conditions[ $loop ] ) {
                                    $none_contained = 'yes';
                                    foreach ( $condition_values as $condition_value ) {
                                        if ( strpos( $uri_for_custom_rules, $condition_value ) !== false ) {
                                            $none_contained = 'no';
                                            break;
                                        }
                                    }
                                    if ( 'yes' === $none_contained ) {
                                        $current_uri_selected = 'yes';
                                        break;
                                    }
                                }

                            // There is only a single value in this condition
                            } else {
                                if ( ( 'contains' === $conditions[ $loop ] && strpos( $uri_for_custom_rules, $values[ $loop ] ) !== false )
                                    || ( 'does-not-contain' === $conditions[ $loop ] && strpos( $uri_for_custom_rules, $values[ $loop ] ) === false ) ) {
                                    $current_uri_selected = 'yes';
                                    break;
                                }
                            }
                        }
                    }

                    /*
                     * If the current URI is selected and we need to deactivate on selected,
                     * or it is not selected and we need to deactivate on all except selected, we deactivate the plugin(s).
                     */
                    if ( ( ( 'selected' === $results['deactivation_type'] && 'yes' === $current_uri_selected )
                        || ( 'except' === $results['deactivation_type'] && 'no' === $current_uri_selected ) )
                        && 'yes' === dppp_custom_function_allows( $results, $custom_function_cache ) ) {

                        // For plugin group rules, we put the plugins in an array and go through them. If it is a single plugin, it works too.
                        $plugin_files = explode( ';;', $results['plugin_file'] );
                        foreach ( $plugin_files as $plugin_file ) {

                            // Currently filtering the option for the sidewide active plugins on a multisite
                            if ( 'site_option_active_sitewide_plugins' === $current_filter ) {
                                if ( array_key_exists( $plugin_file, $active_plugins ) ) {
                                    unset( $active_plugins[ $plugin_file ] );
                                    if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                        $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                        $GLOBALS['dppp_affected_by_custom_rules'] = 'yes';
                                    }
                                }

                            // Currently filtering the option for active plugins on a single site
                            } else {
                                $key = array_search( $plugin_file, $active_plugins );
                                if ( false !== $key ) {
                                    unset( $active_plugins[ $key ] );
                                    if ( ! in_array( $plugin_file, $GLOBALS['dppp_deactivated_plugins'] ) ) {
                                        $GLOBALS['dppp_deactivated_plugins'][] = $plugin_file;
                                        $GLOBALS['dppp_affected_by_custom_rules'] = 'yes';
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // We cache the $active_plugins value for this request before returning it
        if ( 'option_active_plugins' === $current_filter ) {
            $GLOBALS['dppp_cache_option'] = $active_plugins;
        } else {
            $GLOBALS['dppp_cache_site_option'] = $active_plugins;
        }
    }

    return $active_plugins;
}

/**
 * Tells us if the custom function of the rule allows the execution of the rule
 * @param array $results
 * @param array $custom_function_cache
 * @return string
 */
function dppp_custom_function_allows( $results, &$custom_function_cache ) {
    $custom_function_allows = 'yes';
    if ( array_key_exists( 'custom_function', $results ) && ! empty( $results['custom_function'] ) && function_exists( $results['custom_function'] ) ) {
        if ( array_key_exists( $results['custom_function'], $custom_function_cache ) ) {
            $custom_function_result = $custom_function_cache[ $results['custom_function'] ];
        } else {
            $custom_function_result = call_user_func( $results['custom_function'] );
            $custom_function_cache[ $results['custom_function'] ] = $custom_function_result;
        }
        if ( ! is_bool( $custom_function_result ) || true !== $custom_function_result ) {
            $custom_function_allows = "no";
        }
    }
    return $custom_function_allows;
}

/**
 * Based on the user agent it tells if the visitor is using a mobile device or a desktop one.
 * It is similar to the function built in wordpress, but we load it here with another name, so we make sure we can use it so early.
 * Also it detects iPad with Firefox correctly as a mobile device.
 * @return bool
 */
function dppp_is_mobile() {
    if ( empty( $_SERVER['HTTP_USER_AGENT'] ) ) {
        $is_mobile = false;
    } elseif ( strpos( $_SERVER['HTTP_USER_AGENT'], 'Mobile' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'Android' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'Silk/' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'Kindle' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'BlackBerry' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'Opera Mini' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'iPad' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'iPhone' ) !== false
        || strpos( $_SERVER['HTTP_USER_AGENT'], 'Opera Mobi' ) !== false ) {
            $is_mobile = true;
    } else {
        $is_mobile = false;
    }
    return $is_mobile;
}

/**
 * Checks if this is a REST API request
 * @return bool
 */
function dppp_is_rest_request() {
    if ( ( defined( 'REST_REQUEST' ) && REST_REQUEST ) || isset( $_GET['rest_route'] )
        || ( isset( $_SERVER['REQUEST_URI'] ) && strpos( $_SERVER['REQUEST_URI'], '/wp-json/' ) !== false ) ) {
        return true;
    }
    return false;
}

/**
 * Returns the current server URI to use for rules with Page URI selection (we could ignore parameters or not)
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string
 * @return string
 */
if ( ! function_exists( 'dppp_get_uri_for_page_rules' ) ) {
    function dppp_get_uri_for_page_rules( $decoded_uri ) {
        $uri_parameters_page = dppp_get_option( 'dppp-uri-parameters' );
        if ( get_option( 'permalink_structure' ) !== '' && 'ignore-all' === $uri_parameters_page ) {
            return strtok( $decoded_uri, '?' );
        } elseif ( 'ignore-chosen' === $uri_parameters_page ) {
            $chosen_parameters_string = dppp_get_option( 'dppp-chosen-uri-parameters' );
            $chosen_parameters_string = preg_replace( '/\s+/', '', $chosen_parameters_string );
            $chosen_parameters_array = explode( ',', $chosen_parameters_string );
            return rawurldecode( remove_query_arg( $chosen_parameters_array, $decoded_uri ) );
        } else {
            return $decoded_uri;
        }

    }
}

/**
 * Returns the current server URI to use for rules with Custom URI selection (we could ignore parameters or some or none)
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string
 * @return string
 */
if ( ! function_exists( 'dppp_get_uri_for_custom_rules' ) ) {
    function dppp_get_uri_for_custom_rules( $decoded_uri ) {
        $uri_parameters_custom = dppp_get_option( 'dppp-uri-parameters-custom' );
        if ( get_option( 'permalink_structure' ) !== '' && 'ignore-all' === $uri_parameters_custom ) {
            return strtok( $decoded_uri, '?' );
        } elseif ( 'ignore-chosen' === $uri_parameters_custom ) {
            $chosen_parameters_string = dppp_get_option( 'dppp-chosen-uri-parameters' );
            $chosen_parameters_string = preg_replace( '/\s+/', '', $chosen_parameters_string );
            $chosen_parameters_array = explode( ',', $chosen_parameters_string );
            return rawurldecode( remove_query_arg( $chosen_parameters_array, $decoded_uri ) );
        } else {
            return $decoded_uri;
        }
    }
}

/**
 * Returns true if we are on an admin page that we need to not affect by rules.
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @return bool
 */
if ( ! function_exists( 'dppp_skip_this_admin_page' ) ) {
    function dppp_skip_this_admin_page() {
        if ( is_admin() && ( ( isset( $_GET['page'] ) && 'deactivate-plugins-per-page' === $_GET['page'] )
            || ( strpos( $_SERVER['REQUEST_URI'], '/plugins.php' ) !== false && ! isset( $_GET['page'] ) )
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-install.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-editor.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update-core.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/widgets.php?legacy-widget-preview' ) !== false ) ) {
            return true;
        }
        return false;
    }
}

/**
 * Returns true if we are on a network admin page that we need to not affect by rules
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @return bool
 */
if ( ! function_exists( 'dppp_skip_this_network_admin_page' ) ) {
    function dppp_skip_this_network_admin_page() {
        if ( is_network_admin() && ( ( strpos( $_SERVER['REQUEST_URI'], '/plugins.php' ) !== false && ! isset( $_GET['page'] ) )
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-install.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/plugin-editor.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update-core.php' ) !== false
            || strpos( $_SERVER['REQUEST_URI'], '/update.php' ) !== false ) ) {
            return true;
        }
        return false;
    }
}

/**
 * Checks if the string is a valid URL
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @param string $string
 * @return bool
 */
if ( ! function_exists( 'dppp_is_valid_url' ) ) {
    function dppp_is_valid_url( $string ) {
        if ( filter_var( $string, FILTER_VALIDATE_URL ) === false ) {
            return false;
        }
        return true;
    }
}

/**
 * Checks if the current device is a local device (has the cookie for local mode).
 * IMPORTANT: This function is declared in both the normal plugin file and the mu-plugin file. This way we can use it in the mu-plugin before the normal
 * plugin is loaded, and also use it in the normal plugin in case the mu-plugin is missing. When changes are made to it, both places need to be updated.
 * @return bool
 */
if ( ! function_exists( 'dppp_is_local_device' ) ) {
    function dppp_is_local_device() {
        $cookie = dppp_get_option( 'dppp-local-mode-cookie' );
        if ( false !== $cookie && isset( $_COOKIE[ $cookie ] ) && $_COOKIE[ $cookie ] === $cookie ) {
            return true;
        }
        return false;
    }
}
