<?php
if (!defined('ABSPATH')) {
    exit;
}

class BebanStore_Your_Savings extends BebanStore_Feature {
    public function __construct() {
        parent::__construct(
            'your_savings',
            __('سود شما از خرید', 'bebanstore'),
            __('نمایش کل سود و تخفیف کاربر در سبد خرید و صفحه تسویه', 'bebanstore')
        );
    }

    protected function register_hooks() {
        // Add hooks for cart and checkout display
        add_action('woocommerce_cart_totals_after_order_total', array($this, 'display_cart_savings'), 10);
        add_action('woocommerce_review_order_after_order_total', array($this, 'display_checkout_savings'), 10);
        
        // Add hook for order details table
        add_filter('woocommerce_get_order_item_totals', array($this, 'add_savings_to_order_table'), 10, 2);
        
        // Add AJAX handlers
        add_action('wp_ajax_beban_update_cart_savings', array($this, 'ajax_update_cart_savings'));
        add_action('wp_ajax_nopriv_beban_update_cart_savings', array($this, 'ajax_update_cart_savings'));
        
        // Add assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
    }

    /**
     * Calculate total cart savings (product discounts + coupons)
     */
    private function calculate_cart_savings() {
        if (!function_exists('WC') || !WC()->cart) {
            return 0;
        }
        
        $cart = WC()->cart;
        $total_savings = 0;
        
        // Calculate product discounts
        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            $product = $cart_item['data'];
            if ($product) {
                $regular_price = $product->get_regular_price();
                $sale_price = $product->get_sale_price();
                
                if ($regular_price && $sale_price && $regular_price > $sale_price) {
                    $item_savings = ($regular_price - $sale_price) * $cart_item['quantity'];
                    $total_savings += $item_savings;
                }
            }
        }
        
        // Calculate coupon discounts
        $coupon_discount = $cart->get_discount_total();
        $total_savings += $coupon_discount;
        
        return $total_savings;
    }

    /**
     * Display "Your Savings" in the cart
     */
    public function display_cart_savings() {
        $savings = $this->calculate_cart_savings();
        
        if ($savings > 0) {
            $options = get_option('bebanstore_options');
            $feature_options = isset($options['feature_settings']['your_savings']) 
                ? $options['feature_settings']['your_savings'] 
                : array();
            
            // Check if display in cart is enabled
            $show_in_cart = isset($feature_options['show_in_cart']) 
                ? $feature_options['show_in_cart'] 
                : false;
            
            if (!$show_in_cart) {
                return;
            }
            
            $label = isset($feature_options['savings_label']) 
                ? $feature_options['savings_label'] 
                : __('سود شما از خرید:', 'bebanstore');
            
            $show_percentage = isset($feature_options['show_percentage']) 
                ? $feature_options['show_percentage'] 
                : false;
            
            $percentage_text = '';
            if ($show_percentage) {
                $cart_total = WC()->cart->get_cart_contents_total();
                if ($cart_total > 0) {
                    $percentage = round(($savings / ($cart_total + $savings)) * 100, 1);
                    $percentage_text = ' <span class="beban-savings-percentage">(' . $percentage . '%)</span>';
                }
            }
            
            $savings_color = isset($feature_options['savings_color']) 
                ? $feature_options['savings_color'] 
                : '#28a745';
            
            ?>
            <tr class="beban-cart-savings" style="--beban-savings-color: <?php echo esc_attr($savings_color); ?>;">
                <th><?php echo esc_html($label); ?></th>
                <td data-title="<?php echo esc_attr($label); ?>">
                    <span class="beban-savings-amount"><?php echo wc_price($savings); ?></span><?php echo $percentage_text; ?>
                </td>
            </tr>
            <?php
        }
    }

    /**
     * Display "Your Savings" on the checkout page
     */
    public function display_checkout_savings() {
        $savings = $this->calculate_cart_savings();
        
        if ($savings > 0) {
            $options = get_option('bebanstore_options');
            $feature_options = isset($options['feature_settings']['your_savings']) 
                ? $options['feature_settings']['your_savings'] 
                : array();
            
            // Check if display in checkout is enabled
            $show_in_checkout = isset($feature_options['show_in_checkout']) 
                ? $feature_options['show_in_checkout'] 
                : false;
            
            if (!$show_in_checkout) {
                return;
            }
            
            $label = isset($feature_options['savings_label']) 
                ? $feature_options['savings_label'] 
                : __('سود شما از خرید:', 'bebanstore');
            
            $show_percentage = isset($feature_options['show_percentage']) 
                ? $feature_options['show_percentage'] 
                : false;
            
            $percentage_text = '';
            if ($show_percentage) {
                $cart_total = WC()->cart->get_cart_contents_total();
                if ($cart_total > 0) {
                    $percentage = round(($savings / ($cart_total + $savings)) * 100, 1);
                    $percentage_text = ' <span class="beban-savings-percentage">(' . $percentage . '%)</span>';
                }
            }
            
            $savings_color = isset($feature_options['savings_color']) 
                ? $feature_options['savings_color'] 
                : '#28a745';
            
            ?>
            <tr class="beban-checkout-savings" style="--beban-savings-color: <?php echo esc_attr($savings_color); ?>;">
                <th><?php echo esc_html($label); ?></th>
                <td><?php echo wc_price($savings); ?><?php echo $percentage_text; ?></td>
            </tr>
            <?php
        }
    }

    /**
     * Add "Your Savings" to the order details table
     */
    public function add_savings_to_order_table($total_rows, $order) {
        $total_savings = 0;
        
        // Calculate product discounts
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                $regular_price = $product->get_regular_price();
                $sale_price = $product->get_sale_price();
                
                if ($regular_price && $sale_price && $regular_price > $sale_price) {
                    $item_savings = ($regular_price - $sale_price) * $item->get_quantity();
                    $total_savings += $item_savings;
                }
            }
        }
        
        // Add coupon discounts
        $coupon_discount = $order->get_discount_total();
        $total_savings += $coupon_discount;
        
        if ($total_savings > 0) {
            $options = get_option('bebanstore_options');
            $feature_options = isset($options['feature_settings']['your_savings']) 
                ? $options['feature_settings']['your_savings'] 
                : array();
            
            // Check if display in order details is enabled
            $show_in_order_details = isset($feature_options['show_in_order_details']) 
                ? $feature_options['show_in_order_details'] 
                : false;
            
            if ($show_in_order_details) {
                $label = isset($feature_options['savings_label']) 
                    ? $feature_options['savings_label'] 
                    : __('سود شما از خرید:', 'bebanstore');
                
                // Add savings row to the totals table
                $total_rows['savings'] = array(
                    'label' => $label,
                    'value' => wc_price($total_savings)
                );
            }
        }
        
        return $total_rows;
    }

    /**
     * AJAX update for "Your Savings" in the cart
     */
    public function ajax_update_cart_savings() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'beban_savings_nonce')) {
            wp_send_json_error('Invalid nonce');
        }
        
        $savings = $this->calculate_cart_savings();
        $savings_html = '';
        
        if ($savings > 0) {
            $options = get_option('bebanstore_options');
            $feature_options = isset($options['feature_settings']['your_savings']) 
                ? $options['feature_settings']['your_savings'] 
                : array();
            
            // Check if display in cart is enabled for AJAX
            $show_in_cart = isset($feature_options['show_in_cart']) 
                ? $feature_options['show_in_cart'] 
                : false;
            
            if (!$show_in_cart) {
                wp_send_json_success(array(
                    'savings_html' => '',
                    'savings_amount' => 0
                ));
                return;
            }
            
            $label = isset($feature_options['savings_label']) 
                ? $feature_options['savings_label'] 
                : __('سود شما:', 'bebanstore');
            
            $show_percentage = isset($feature_options['show_percentage']) 
                ? $feature_options['show_percentage'] 
                : false;
            
            $percentage_text = '';
            if ($show_percentage) {
                $cart_total = WC()->cart->get_cart_contents_total();
                if ($cart_total > 0) {
                    $percentage = round(($savings / ($cart_total + $savings)) * 100, 1);
                    $percentage_text = ' <span class="beban-savings-percentage">(' . $percentage . '%)</span>';
                }
            }
            
            $savings_color = isset($feature_options['savings_color']) 
                ? $feature_options['savings_color'] 
                : '#28a745';
            
            $savings_html = '
            <tr class="beban-cart-savings" style="--beban-savings-color: ' . esc_attr($savings_color) . ';">
                <th>' . esc_html($label) . '</th>
                <td data-title="' . esc_attr($label) . '">
                    <span class="beban-savings-amount">' . wc_price($savings) . '</span>' . $percentage_text . '
                </td>
            </tr>';
        }
        
        wp_send_json_success(array(
            'savings_html' => $savings_html,
            'savings_amount' => $savings
        ));
    }

    /**
     * Enqueue assets
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'bebanstore-your-savings',
            BEBANSTORE_PLUGIN_URL . 'assets/css/your-savings.css',
            array(),
            BEBANSTORE_VERSION
        );
        
        wp_enqueue_script(
            'bebanstore-your-savings',
            BEBANSTORE_PLUGIN_URL . 'assets/js/your-savings.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );
        
        wp_localize_script('bebanstore-your-savings', 'beban_savings_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('beban_savings_nonce')
        ));
    }
} 