<?php
/**
 * Tracking Code Feature
 * 
 * @package BebanStore
 * @since 1.0.0
 */
class BebanStore_Tracking_Code extends BebanStore_Feature {
    /**
     * Initialize the feature
     */
    public function __construct() {
        parent::__construct(
            'tracking_code',
            __('کد رهگیری پستی', 'bebanstore'),
            __('نمایش کد رهگیری پستی در صفحه جزئیات سفارش', 'bebanstore')
        );

        // Register assets
        $this->css_files = array(
            'tracking-code' => 'tracking-code.css'
        );
        $this->js_files = array(
            'tracking-code' => 'tracking-code.js'
        );
    }

    /**
     * Register hooks
     */
    protected function register_hooks() {
        if (!$this->is_enabled()) {
            return;
        }

        // Add meta box to order page
        add_action('add_meta_boxes', array($this, 'add_tracking_code_meta_box'));

        // Save tracking code - support both old and new WooCommerce
        add_action('woocommerce_process_shop_order_meta', array($this, 'save_tracking_code'));
        add_action('woocommerce_order_object_updated_props', array($this, 'save_tracking_code_new'), 10, 2);

        // Display tracking code based on user settings
        $this->add_tracking_code_hooks();

        // Register and enqueue assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_feature_assets'));
    }

    /**
     * Add tracking code hooks based on user settings
     */
    private function add_tracking_code_hooks() {
        $settings = $this->get_options();
        $display_position = isset($settings['display_position']) ? $settings['display_position'] : 'before_order_table';
        $show_in_thankyou = isset($settings['show_in_thankyou']) ? $settings['show_in_thankyou'] : false;
        
        // Map position to WooCommerce hooks
        $hook_map = array(
            'before_order_table' => 'woocommerce_order_details_before_order_table',
            'after_order_table' => 'woocommerce_order_details_after_order_table',
            'after_customer_details' => 'woocommerce_order_details_after_customer_details'
        );
        
        // Add hook for selected position with appropriate priority
        if (isset($hook_map[$display_position])) {
            $priority = ($display_position === 'after_order_table') ? 1 : 10; // Show before title (priority 1)
            add_action($hook_map[$display_position], array($this, 'display_tracking_code'), $priority);
        }
        
        // Add hook for thank you page if enabled
        if ($show_in_thankyou) {
            add_action('woocommerce_thankyou', array($this, 'display_tracking_code'));
        }
    }

    /**
     * Add meta box to order page
     */
    public function add_tracking_code_meta_box() {
        // Support both old and new WooCommerce order post types
        $post_types = array('shop_order');
        
        // Add support for HPOS (High-Performance Order Storage) if available
        if (class_exists('Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController')) {
            $post_types[] = 'woocommerce_page_wc-orders';
        }
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'beban_tracking_code',
                __('کد رهگیری پستی', 'bebanstore'),
                array($this, 'render_tracking_code_meta_box'),
                $post_type,
                'side',
                'high'
            );
        }
    }

    /**
     * Render tracking code meta box
     */
    public function render_tracking_code_meta_box($post) {
        // Get order ID safely
        $order_id = is_object($post) ? $post->ID : $post;
        $order = wc_get_order($order_id);
        
        if (!$order) {
            echo '<p>' . __('خطا در بارگذاری سفارش', 'bebanstore') . '</p>';
            return;
        }
        
        $tracking_code = get_post_meta($order->get_id(), '_beban_tracking_code', true);
        $postal_center = get_post_meta($order->get_id(), '_beban_postal_center', true);
        $available_centers = $this->get_postal_centers();
        ?>
        <p>
            <label for="_beban_tracking_code"><?php _e('کد رهگیری پستی', 'bebanstore'); ?></label>
            <input type="text" 
                   id="_beban_tracking_code" 
                   name="_beban_tracking_code" 
                   value="<?php echo esc_attr($tracking_code); ?>" 
                   class="widefat" 
                   placeholder="<?php _e('مثال: 948205173624019385672413', 'bebanstore'); ?>">
            <span class="description"><?php _e('کد رهگیری پستی برای این سفارش', 'bebanstore'); ?></span>
        </p>
        
        <?php if (!empty($available_centers)): ?>
        <p>
            <label for="_beban_postal_center"><?php _e('مرکز پستی', 'bebanstore'); ?></label>
            <select id="_beban_postal_center" name="_beban_postal_center" class="widefat">
                <option value=""><?php _e('انتخاب مرکز پستی', 'bebanstore'); ?></option>
                <?php foreach ($available_centers as $center_id => $center_name): ?>
                    <option value="<?php echo esc_attr($center_id); ?>" <?php selected($postal_center, $center_id); ?>>
                        <?php echo esc_html($center_name); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <span class="description"><?php _e('مرکز پستی که این سفارش از طریق آن ارسال شده است', 'bebanstore'); ?></span>
        </p>
        <?php endif; ?>
        <?php
    }

    /**
     * Save tracking code (old WooCommerce)
     */
    public function save_tracking_code($order_id) {
        if (isset($_POST['_beban_tracking_code'])) {
            update_post_meta($order_id, '_beban_tracking_code', sanitize_text_field($_POST['_beban_tracking_code']));
        }
        if (isset($_POST['_beban_postal_center'])) {
            update_post_meta($order_id, '_beban_postal_center', sanitize_text_field($_POST['_beban_postal_center']));
        }
    }

    /**
     * Save tracking code (new WooCommerce with HPOS)
     */
    public function save_tracking_code_new($order, $updated_props) {
        if (isset($_POST['_beban_tracking_code'])) {
            $order->update_meta_data('_beban_tracking_code', sanitize_text_field($_POST['_beban_tracking_code']));
        }
        if (isset($_POST['_beban_postal_center'])) {
            $order->update_meta_data('_beban_postal_center', sanitize_text_field($_POST['_beban_postal_center']));
        }
    }

    /**
     * Get feature options
     * 
     * @return array Feature options
     */
    public function get_options() {
        $options = get_option('bebanstore_options', array());
        $feature_settings = isset($options['feature_settings']['tracking_code']) ? $options['feature_settings']['tracking_code'] : array();
        
        return array(
            'tracking_box_bg' => isset($feature_settings['box_bg_color']) ? $feature_settings['box_bg_color'] : '#f8f9fa',
            'tracking_text_color' => isset($feature_settings['text_color']) ? $feature_settings['text_color'] : '#212529',
            'tracking_link_color' => isset($feature_settings['link_color']) ? $feature_settings['link_color'] : '#0d6efd',
            'tracking_code_color' => isset($feature_settings['code_color']) ? $feature_settings['code_color'] : '#292D32',
            'tracking_tooltip_bg' => isset($feature_settings['tooltip_bg_color']) ? $feature_settings['tooltip_bg_color'] : '#212529',
            'tracking_tooltip_text' => isset($feature_settings['tooltip_text_color']) ? $feature_settings['tooltip_text_color'] : '#ffffff',
            'info_text' => isset($feature_settings['info_text']) ? $feature_settings['info_text'] : 'با استفاده از سامانه رهگیری پست می‌توانید از وضعیت مرسوله باخبر شوید.',
            'tracking_link_text' => isset($feature_settings['tracking_link_text']) ? $feature_settings['tracking_link_text'] : 'ورود به سامانه رهگیری',
            'code_label' => isset($feature_settings['code_label']) ? $feature_settings['code_label'] : 'کد رهگیری:',
            'display_position' => isset($feature_settings['display_position']) ? $feature_settings['display_position'] : 'before_order_table',
            'show_in_thankyou' => isset($feature_settings['show_in_thankyou']) ? $feature_settings['show_in_thankyou'] : false,
            'postal_centers' => $this->get_active_postal_centers($feature_settings),
            'show_delivery_type' => isset($feature_settings['show_delivery_type']) ? $feature_settings['show_delivery_type'] : true,
            'delivery_type_position' => isset($feature_settings['delivery_type_position']) ? $feature_settings['delivery_type_position'] : 'separate_label'
        );
    }

    /**
     * Get active postal centers from settings
     * 
     * @param array $feature_settings Feature settings
     * @return array Active postal centers
     */
    private function get_active_postal_centers($feature_settings) {
        $active_centers = array();
        
        $center_mapping = array(
            'postal_center_post_ir' => 'post_ir',
            'postal_center_tipax' => 'tipax',
            'postal_center_deka_post' => 'deka_post',
            'postal_center_custom' => 'custom'
        );
        
        foreach ($center_mapping as $setting_key => $center_id) {
            if (isset($feature_settings[$setting_key]) && $feature_settings[$setting_key]) {
                $active_centers[] = $center_id;
            }
        }
        
        // Default to post_ir if no centers are selected
        if (empty($active_centers)) {
            $active_centers = array('post_ir');
        }
        
        return $active_centers;
    }

    /**
     * Get available postal centers
     * 
     * @return array Available postal centers
     */
    public function get_postal_centers() {
        $options = $this->get_options();
        $active_centers = $options['postal_centers'];
        
        $all_centers = array(
            'post_ir' => __('اداره پست', 'bebanstore'),
            'tipax' => __('تیپاکس', 'bebanstore'),
            'deka_post' => __('دکا پست', 'bebanstore'),
            'custom' => __('سایر مراکز پستی', 'bebanstore')
        );
        
        $available_centers = array();
        foreach ($active_centers as $center_id) {
            if (isset($all_centers[$center_id])) {
                $available_centers[$center_id] = $all_centers[$center_id];
            }
        }
        
        return $available_centers;
    }

    /**
     * Get tracking link text based on postal center
     * 
     * @param string $center_id Postal center ID
     * @param string $base_text Base link text
     * @return string Modified link text
     */
    public function get_tracking_link_text($center_id, $base_text) {
        $center_names = array(
            'post_ir' => 'پست',
            'tipax' => 'تیپاکس',
            'deka_post' => 'دکا پست',
            'custom' => ''
        );
        
        if (isset($center_names[$center_id]) && !empty($center_names[$center_id])) {
            return $base_text . ' ' . $center_names[$center_id];
        }
        
        return $base_text;
    }

    /**
     * Get tracking URL for postal center
     * 
     * @param string $center_id Postal center ID
     * @param string $tracking_code Tracking code
     * @return string Tracking URL
     */
    public function get_tracking_url($center_id, $tracking_code) {
        $urls = array(
            'post_ir' => 'https://tracking.post.ir/search.aspx?id=' . $tracking_code,
            'tipax' => 'https://tipaxco.com/tracking',
            'deka_post' => 'https://dekapost.com/',
            'custom' => '#' // Will be handled separately
        );
        
        return isset($urls[$center_id]) ? $urls[$center_id] : '#';
    }

    /**
     * Display tracking code in order details
     */
    public function display_tracking_code($order) {
        // Prevent duplicate display
        static $displayed = false;
        if ($displayed) {
            return;
        }
        
        // Get order ID
        $order_id = is_object($order) ? $order->get_id() : $order;
        if (!$order_id) {
            return;
        }
        
        $tracking_code = get_post_meta($order_id, '_beban_tracking_code', true);
        if (!$tracking_code) {
            return;
        }

        $postal_center = get_post_meta($order_id, '_beban_postal_center', true);
        $available_centers = $this->get_postal_centers();
        $center_name = isset($available_centers[$postal_center]) ? $available_centers[$postal_center] : '';
        
        $tracking_url = $this->get_tracking_url($postal_center, $tracking_code);
        $options = $this->get_options();
        
        // Get link text based on settings
        $link_text = $options['tracking_link_text'];
        if ($options['delivery_type_position'] === 'in_link_text' && !empty($postal_center)) {
            $link_text = $this->get_tracking_link_text($postal_center, $link_text);
        }
        ?>
        <div class="beban-tracking-box" style="background-color: <?php echo esc_attr($options['tracking_box_bg']); ?>">
            <p class="beban-tracking-text" style="color: <?php echo esc_attr($options['tracking_text_color']); ?>">
                <?php echo esc_html($options['info_text']); ?>
            </p>
            <?php if ($tracking_url !== '#' && !empty($postal_center) && $postal_center !== 'custom'): ?>
            <p class="beban-tracking-link">
                <a href="<?php echo esc_url($tracking_url); ?>" target="_blank" style="color: <?php echo esc_attr($options['tracking_link_color']); ?>">
                    <?php echo esc_html($link_text); ?>
                    <svg style="width: 20px; height: 20px; fill: <?php echo esc_attr($options['tracking_link_color']); ?>; vertical-align: middle; margin-left: 5px;">
                        <use xlink:href="#chevronLeft"></use>
                    </svg>
                </a>
            </p>
            <?php endif; ?>
            <p class="beban-tracking-code" style="color: <?php echo esc_attr($options['tracking_code_color']); ?>">
                <?php echo esc_html($options['code_label']); ?> 
                <span class="beban-copy-code" data-code="<?php echo esc_attr($tracking_code); ?>">
                    <?php echo esc_html($tracking_code); ?>
                    <svg class="beban-copy-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="stroke: <?php echo esc_attr($options['tracking_code_color']); ?>">
                        <path d="M16 12.9V17.1C16 20.6 14.6 22 11.1 22H6.9C3.4 22 2 20.6 2 17.1V12.9C2 9.4 3.4 8 6.9 8H11.1C14.6 8 16 9.4 16 12.9Z" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                        <path d="M22 6.9V11.1C22 14.6 20.6 16 17.1 16H16V12.9C16 9.4 14.6 8 11.1 8H8V6.9C8 3.4 9.4 2 12.9 2H17.1C20.6 2 22 3.4 22 6.9Z" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                    </svg>
                </span>
            </p>
            <?php if ($options['show_delivery_type'] && $options['delivery_type_position'] === 'separate_label' && !empty($center_name)): ?>
            <p class="beban-tracking-center" style="color: <?php echo esc_attr($options['tracking_text_color']); ?>; font-size: 14px; margin-top: 8px;">
                <strong><?php _e('نوع ارسال:', 'bebanstore'); ?></strong> <?php echo esc_html($center_name); ?>
            </p>
            <?php endif; ?>
        </div>
        <?php
        $displayed = true;
    }

    /**
     * Enqueue feature assets
     */
    public function enqueue_feature_assets() {
        // Enqueue CSS only on order pages
        if (is_wc_endpoint_url('order-received') || is_wc_endpoint_url('view-order')) {
            wp_enqueue_style(
                'bebanstore-tracking-code',
                BEBANSTORE_PLUGIN_URL . 'assets/css/tracking-code.css',
                array(),
                BEBANSTORE_VERSION
            );
        }

        // Enqueue JS on all pages
        wp_enqueue_script(
            'bebanstore-tracking-code',
            BEBANSTORE_PLUGIN_URL . 'assets/js/tracking-code.js',
            array('jquery'),
            BEBANSTORE_VERSION,
            true
        );
    }

    /**
     * Cleanup feature resources
     */
    public function cleanup() {
        remove_action('add_meta_boxes', array($this, 'add_tracking_code_meta_box'));
        remove_action('woocommerce_process_shop_order_meta', array($this, 'save_tracking_code'));
        
        // Remove all possible tracking code hooks
        $hooks = array(
            'woocommerce_order_details_before_order_table',
            'woocommerce_order_details_after_order_table',
            'woocommerce_order_details_after_customer_details',
            'woocommerce_thankyou'
        );
        
        foreach ($hooks as $hook) {
            remove_action($hook, array($this, 'display_tracking_code'));
        }
        
        remove_action('wp_enqueue_scripts', array($this, 'enqueue_feature_assets'));
    }
} 