/**
 * BebanStore Wishlist JavaScript
 * Handles wishlist functionality with AJAX
 */

jQuery(document).ready(function($) {

    // Apply product content styles from data attributes
    function applyProductContentStyles() {
        $('.beban-price-wrapper').each(function() {
            const $wrapper = $(this);
            const regularPriceSize = $wrapper.data('regular-price-size');
            const regularPriceColor = $wrapper.data('regular-price-color');
            const salePriceSize = $wrapper.data('sale-price-size');
            const salePriceColor = $wrapper.data('sale-price-color');
            
            // Apply styles to del (regular price)
            if (regularPriceSize || regularPriceColor) {
                $wrapper.find('del').css({
                    'font-size': regularPriceSize ? regularPriceSize + 'px' : '',
                    'color': regularPriceColor || ''
                });
            }
            
            // Apply styles to ins (sale price)
            if (salePriceSize || salePriceColor) {
                $wrapper.find('ins').css({
                    'font-size': salePriceSize ? salePriceSize + 'px' : '',
                    'color': salePriceColor || ''
                });
            }
        });
    }
    
    // Apply styles on page load
    applyProductContentStyles();

    // Bulk Actions functionality
    function initBulkActions() {
        const $selectAllCheckbox = $('#beban-wishlist-select-all');
        const $productCheckboxes = $('.beban-wishlist-product-checkbox');
        const $bulkRemoveBtn = $('.beban-wishlist-bulk-remove-btn');
        const $selectedCount = $('.beban-wishlist-selected-count');
        

        
        // Handle select all checkbox
        $selectAllCheckbox.on('change', function() {
            const isChecked = $(this).is(':checked');
            $productCheckboxes.prop('checked', isChecked);
            updateBulkRemoveButton();
        });
        
        // Handle individual product checkboxes
        $productCheckboxes.on('change', function() {
            updateSelectAllCheckbox();
            updateBulkRemoveButton();
        });
        
        // Handle bulk remove button
        $bulkRemoveBtn.on('click', function() {
            const selectedProducts = $productCheckboxes.filter(':checked');
            
            if (selectedProducts.length === 0) {
                return;
            }
            
            // Show confirmation if enabled
            if (typeof bebanWishlistAjax !== 'undefined' && bebanWishlistAjax.bulk_remove_confirmation) {
                if (!confirm('آیا از حذف محصولات انتخاب شده اطمینان دارید؟')) {
                    return;
                }
            }
            
            const $btn = $(this);
            $btn.prop('disabled', true);
            
            // Show loader on the products container
            const $productsContainer = $('.beban-wishlist-grid, .beban-wishlist-list').first();
            
            if (typeof BebanStoreLoader !== 'undefined' && $productsContainer.length > 0) {
                const animationType = bebanWishlistAjax.animation_type || 'dots';
                const overlayColor = bebanWishlistAjax.remove_loader_overlay_color || '#ffffff';
                const animationColor = bebanWishlistAjax.remove_loader_animation_color || '#81858B';
                
                if (animationType === 'none') {
                    // بدون انیمیشن: فقط غیرفعال کردن محصولات
                    $productsContainer.addClass('beban-disabled');
                } else {
                    // انیمیشن‌های بصری: spinner, dots
                    BebanStoreLoader.showLoaderWithCustomColors($productsContainer[0], animationType, animationColor, overlayColor);
                }
            }
            
            let completed = 0;
            const total = selectedProducts.length;
            
            selectedProducts.each(function() {
                const productId = $(this).val();
                
                $.ajax({
                    url: bebanWishlistAjax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'beban_handle_wishlist',
                        product_id: productId,
                        action_type: 'remove',
                        nonce: bebanWishlistAjax.nonce
                    },
                    success: function(response) {
                        completed++;
                        
                        if (completed === total) {
                            // Refresh the page after all removals
                            location.reload();
                        }
                    },
                    error: function() {
                        completed++;
                        
                        if (completed === total) {
                            // Hide loader on error
                            if (typeof BebanStoreLoader !== 'undefined' && $productsContainer.length > 0) {
                                const animationType = bebanWishlistAjax.animation_type || 'dots';
                                
                                if (animationType === 'none') {
                                    $productsContainer.removeClass('beban-disabled');
                                } else {
                                    BebanStoreLoader.hideLoader($productsContainer[0]);
                                }
                            }
                            
                            $btn.prop('disabled', false);
                            
                            if (typeof BebanStoreToast !== 'undefined') {
                                BebanStoreToast.error('خطا در حذف برخی محصولات');
                            } else {
                                alert('خطا در حذف برخی محصولات');
                            }
                        }
                    }
                });
            });
        });
        
        function updateSelectAllCheckbox() {
            const totalCheckboxes = $productCheckboxes.length;
            const checkedCheckboxes = $productCheckboxes.filter(':checked').length;
            
            if (checkedCheckboxes === 0) {
                $selectAllCheckbox.prop('checked', false).prop('indeterminate', false);
            } else if (checkedCheckboxes === totalCheckboxes) {
                $selectAllCheckbox.prop('checked', true).prop('indeterminate', false);
            } else {
                $selectAllCheckbox.prop('checked', false).prop('indeterminate', true);
            }
        }
        
        function updateBulkRemoveButton() {
            const selectedCount = $productCheckboxes.filter(':checked').length;
            
            if (selectedCount > 0) {
                $bulkRemoveBtn.prop('disabled', false);
                $selectedCount.text(`(${selectedCount})`);
            } else {
                $bulkRemoveBtn.prop('disabled', true);
                $selectedCount.text('(0)');
            }
        }
        

    }

    // Initialize bulk actions if elements exist
    if ($('.beban-wishlist-bulk-actions').length > 0) {
        initBulkActions();
    }


    // Handle wishlist button clicks - handle clicks on button and its children
    $(document).on('click', '.beban-wishlist-btn, .beban-wishlist-btn *', function(e) {
        e.preventDefault();
        e.stopPropagation();

        // Find the actual button (in case we clicked on a child element)
        var $btn = $(this).closest('.beban-wishlist-btn');
        if (!$btn.length) {
            return;
        }
        
        if ($btn.hasClass('beban-disabled')) {
            return;
        }

        // Check stock status before proceeding
        var stockStatus = $btn.data('stock-status');
        var isInStock = $btn.data('in-stock');
        
        if (stockStatus === 'outofstock' || isInStock === false) {
            if (typeof BebanStoreToast !== 'undefined') {
                BebanStoreToast.error('این محصول در حال حاضر ناموجود است.');
            }
            return;
        }

        var uid = parseInt(bebanWishlistAjax.user_id, 10);
        if (uid === 0) {
            if (typeof BebanStoreToast !== 'undefined') {
                BebanStoreToast.error(bebanWishlistAjax.login_message);
            }
            return;
        }

        // Show loader using BebanStoreLoader with custom color and animation type
        if (typeof BebanStoreLoader !== 'undefined') {
            const animationType = bebanWishlistAjax.animation_type || 'dots';
            
            // ذخیره HTML اصلی دکمه برای بازگرداندن
            if (!$btn.data('original-html')) {
                $btn.data('original-html', $btn.html());
            }
            
            if (animationType === 'none') {
                // بدون انیمیشن: فقط غیرفعال کردن دکمه
                $btn.addClass('beban-disabled');
            } else {
                // انیمیشن‌های بصری: spinner, dots
                // مخفی کردن قلب اصلی قبل از نمایش loader
                $btn.find('svg').css('opacity', '0');
                
                // استفاده از رنگ‌های مخصوص دکمه قلب علاقه‌مندی - حالت لودینگ
                const animationColor = bebanWishlistAjax.wishlist_btn_loader_animation_color || '#292D32';
                
                // بدون overlay background برای دکمه قلب
                BebanStoreLoader.showLoaderWithCustomColors($btn[0], animationType, animationColor, null);
            }
        } else {
            $btn.addClass('beban-disabled');
        }

        var product_id = $btn.data('product-id');
        var is_wishlisted = $btn.hasClass('beban-wishlisted');
        var action_type = is_wishlisted ? 'remove' : 'add';

        $.ajax({
            url: bebanWishlistAjax.ajax_url,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'beban_handle_wishlist',
                product_id: product_id,
                action_type: action_type,
                nonce: bebanWishlistAjax.nonce
            },
            success: function(response) {
                // Hide loader first and restore button state
                if (typeof BebanStoreLoader !== 'undefined') {
                    const animationType = bebanWishlistAjax.animation_type || 'dots';
                    
                    if (animationType === 'none') {
                        // بدون انیمیشن: فقط فعال کردن دکمه
                        $btn.removeClass('beban-disabled');
                    } else {
                        // پنهان کردن loader برای انیمیشن‌های بصری
                        BebanStoreLoader.hideLoader($btn[0]);
                    }
                }
                
                // Update all wishlist buttons for this product
                var $allProductButtons = $('.beban-wishlist-btn[data-product-id="' + product_id + '"]');
                
                // بازگرداندن قلب اصلی (opacity) قبل از تغییر SVG
                $allProductButtons.find('svg').css('opacity', '1');
                
                if (response.success) {
                    if (response.data.status === 'added') {
                        $allProductButtons.addClass('beban-wishlisted');
                        $allProductButtons.find('svg').remove();
                        
                        // استفاده از رنگ‌های تنظیمات برای حالت فعال
                        const activeFill = bebanWishlistAjax.wishlist_btn_active_fill || '#292D32';
                        const activeStroke = bebanWishlistAjax.wishlist_btn_active_stroke || '#292D32';
                        
                        // ساخت SVG
                        const svgContent = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">' +
                            '<path d="M12.62 20.8101C12.28 20.9301 11.72 20.9301 11.38 20.8101C8.48 19.8201 2 15.6901 2 8.6901C2 5.6001 4.49 3.1001 7.56 3.1001C9.38 3.1001 10.99 3.9801 12 5.3401C13.01 3.9801 14.63 3.1001 16.44 3.1001C19.51 3.1001 22 5.6001 22 8.6901C22 15.6901 15.52 19.8201 12.62 20.8101Z" fill="' + activeFill + '" stroke="' + activeStroke + '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>' +
                            '</svg>';
                        
                        $allProductButtons.append(svgContent);
                        
                        // Remove loading class and re-enable buttons
                        $allProductButtons.removeClass('bebanstore-loading beban-disabled');
                        
                        // Force enable pointer events and cursor
                        $allProductButtons.css({
                            'pointer-events': 'auto',
                            'cursor': 'pointer'
                        });
                        
                        if (typeof BebanStoreToast !== 'undefined') {
                            BebanStoreToast.success(response.data.message);
                        }
                    } else if (response.data.status === 'removed') {
                        $allProductButtons.removeClass('beban-wishlisted');
                        $allProductButtons.find('svg').remove();
                        
                        // استفاده از رنگ‌های تنظیمات برای حالت عادی
                        const normalFill = bebanWishlistAjax.wishlist_btn_normal_fill || 'none';
                        const normalStroke = bebanWishlistAjax.wishlist_btn_normal_stroke || '#A9ABAD';
                        
                        // ساخت SVG
                        const svgContent = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">' +
                            '<path d="M12.62 20.8101C12.28 20.9301 11.72 20.9301 11.38 20.8101C8.48 19.8201 2 15.6901 2 8.6901C2 5.6001 4.49 3.1001 7.56 3.1001C9.38 3.1001 10.99 3.9801 12 5.3401C13.01 3.9801 14.63 3.1001 16.44 3.1001C19.51 3.1001 22 5.6001 22 8.6901C22 15.6901 15.52 19.8201 12.62 20.8101Z" fill="' + normalFill + '" stroke="' + normalStroke + '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>' +
                            '</svg>';
                        
                        $allProductButtons.append(svgContent);
                        
                        // Remove loading class and re-enable buttons
                        $allProductButtons.removeClass('bebanstore-loading beban-disabled');
                        
                        // Force enable pointer events and cursor
                        $allProductButtons.css({
                            'pointer-events': 'auto',
                            'cursor': 'pointer'
                        });
                        
                        if (typeof BebanStoreToast !== 'undefined') {
                            BebanStoreToast.warning(response.data.message);
                        }
                    } else {
                        // Remove loading class and re-enable buttons
                        $allProductButtons.removeClass('bebanstore-loading beban-disabled');
                        
                        if (typeof BebanStoreToast !== 'undefined') {
                            BebanStoreToast.info(response.data.message);
                        }
                    }
                } else {
                    // Handle error response
                    $allProductButtons.removeClass('bebanstore-loading beban-disabled');
                    
                    if (typeof BebanStoreToast !== 'undefined') {
                        BebanStoreToast.error(response.data.message || 'خطایی رخ داد. لطفاً دوباره امتحان کنید.');
                    }
                }
            },
            error: function(xhr, status, error) {
                if (typeof BebanStoreToast !== 'undefined') {
                    BebanStoreToast.error('خطایی رخ داد: ' + error);
                }
            },
            complete: function() {
                // Re-enable all buttons after AJAX completes
                var $allProductButtons = $('.beban-wishlist-btn[data-product-id="' + product_id + '"]');
                $allProductButtons.removeClass('beban-disabled');
                
                // Remove disabled class if loader is not available
                if (typeof BebanStoreLoader === 'undefined') {
                    $btn.removeClass('beban-disabled');
                }
            }
        });
    });
    
    // Handle add to cart button in wishlist products page
    $(document).on('click', '.beban-wishlist-add-to-cart-btn', function(e) {
        e.preventDefault();
        
        var $btn = $(this);
        if ($btn.hasClass('beban-disabled')) {
            return;
        }
        
        var product_id = $btn.data('product-id');
        
        // Show loader with custom color and animation type
        if (typeof BebanStoreLoader !== 'undefined') {
            const animationType = bebanWishlistAjax.animation_type || 'dots';
            const overlayColor = bebanWishlistAjax.add_to_cart_loader_overlay_color || 'rgba(255, 255, 255, 1)';
            const animationColor = bebanWishlistAjax.add_to_cart_loader_animation_color || '#EF4056';
            
            if (animationType === 'none') {
                // بدون انیمیشن: فقط غیرفعال کردن دکمه
                $btn.addClass('beban-disabled');
            } else {
                // انیمیشن‌های بصری: spinner, dots, text
                const customText = animationType === 'text' ? bebanWishlistAjax.i18n.adding : null;
                BebanStoreLoader.showLoaderWithCustomColors($btn[0], animationType, animationColor, overlayColor, customText);
            }
        } else {
            $btn.addClass('beban-disabled');
        }
        
        // Add to cart using BebanStore AJAX
        $.ajax({
            url: bebanWishlistAjax.ajax_url,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'bebanstore_add_to_cart_from_wishlist',
                product_id: product_id,
                quantity: 1,
                nonce: bebanWishlistAjax.nonce
            },
            success: function(response) {
                // Hide loader
                if (typeof BebanStoreLoader !== 'undefined') {
                    BebanStoreLoader.hideLoader($btn[0]);
                }
                
                if (response.success) {
                    if (typeof BebanStoreToast !== 'undefined') {
                        BebanStoreToast.success(response.data.message || 'محصول به سبد خرید اضافه شد.');
                    }
                    
                    // Update cart count if available
                    if (response.data && response.data.cart_count) {
                        $('.cart-count').text(response.data.cart_count);
                    }
                    
                    // Remove product from wishlist UI if it was removed from wishlist
                    if (response.data && response.data.removed_from_wishlist) {
                        // Remove the entire product card from the list with animation
                        $btn.closest('.beban-wishlist-item').fadeOut(300, function() {
                            $(this).remove();
                            
                            // Check if no products left
                            if ($('.beban-wishlist-item').length === 0) {
                                $('.beban-wishlist-products').html(
                                    '<div style="text-align:center; margin-top:50px;">' +
                                    '<svg width="50" height="50" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="display:block; margin:0 auto 12px;">' +
                                    '<path d="M6.10999 17.5C3.89999 15.43 2 12.48 2 8.67999C2 5.58999 4.49 3.09003 7.56 3.09003C9.38 3.09003 10.99 3.97002 12 5.33002C13.01 3.97002 14.63 3.09003 16.44 3.09003C17.59 3.09003 18.66 3.43999 19.55 4.04999" stroke="rgba(41,45,50,0.3)" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>' +
                                    '<path d="M21.74 7C21.91 7.53 22 8.1 22 8.69C22 15.69 15.52 19.82 12.62 20.82C12.28 20.94 11.72 20.94 11.38 20.82C10.73 20.6 9.91002 20.22 9.02002 19.69" stroke="rgba(41,45,50,0.3)" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>' +
                                    '<path d="M22 2L2 22" stroke="#999" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>' +
                                    '</svg>' +
                                    '<p style="color:#999;">' + bebanWishlistAjax.empty_message + '</p>' +
                                    '</div>'
                                );
                            }
                        });
                        
                        // Update all wishlist buttons for this product across all pages
                        var $allProductButtons = $('.beban-wishlist-btn[data-product-id="' + product_id + '"]');
                        $allProductButtons.removeClass('beban-wishlisted');
                        $allProductButtons.find('svg').remove();
                        
                        // استفاده از رنگ‌های تنظیمات برای حالت عادی
                        const normalFill = bebanWishlistAjax.wishlist_btn_normal_fill || 'none';
                        const normalStroke = bebanWishlistAjax.wishlist_btn_normal_stroke || '#A9ABAD';
                        
                        // ساخت SVG
                        const svgContent = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">' +
                            '<path d="M12.62 20.8101C12.28 20.9301 11.72 20.9301 11.38 20.8101C8.48 19.8201 2 15.6901 2 8.6901C2 5.6001 4.49 3.1001 7.56 3.1001C9.38 3.1001 10.99 3.9801 12 5.3401C13.01 3.9801 14.63 3.1001 16.44 3.1001C19.51 3.1001 22 5.6001 22 8.6901C22 15.6901 15.52 19.8201 12.62 20.8101Z" fill="' + normalFill + '" stroke="' + normalStroke + '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>' +
                            '</svg>';
                        
                        $allProductButtons.append(svgContent);
                    }
                } else {
                    if (typeof BebanStoreToast !== 'undefined') {
                        BebanStoreToast.error(response.data.message || 'خطا در افزودن به سبد خرید.');
                    }
                }
            },
            error: function() {
                // Hide loader and restore button state
                if (typeof BebanStoreLoader !== 'undefined') {
                    const animationType = bebanWishlistAjax.animation_type || 'dots';
                    
                    if (animationType === 'none') {
                        // بدون انیمیشن: فقط فعال کردن دکمه
                        $btn.removeClass('beban-disabled');
                    } else {
                        // پنهان کردن loader برای انیمیشن‌های بصری
                        BebanStoreLoader.hideLoader($btn[0]);
                    }
                }
                
                if (typeof BebanStoreToast !== 'undefined') {
                    BebanStoreToast.error('خطا در افزودن به سبد خرید.');
                }
            },
            complete: function() {
                // Hide loader and restore button state
                if (typeof BebanStoreLoader !== 'undefined') {
                    BebanStoreLoader.hideLoader($btn[0]);
                } else {
                    $btn.removeClass('beban-disabled');
                }
            }
        });
    });

    // Handle remove from wishlist button in wishlist products page
    $(document).on('click', '.beban-wishlist-remove-btn', function(e) {
        e.preventDefault();
        
        var $btn = $(this);
        if ($btn.hasClass('beban-disabled')) {
            return;
        }
        
        var uid = parseInt(bebanWishlistAjax.user_id, 10);
        if (uid === 0) {
            if (typeof BebanStoreToast !== 'undefined') {
                BebanStoreToast.error(bebanWishlistAjax.login_message);
            }
            return;
        }
        
        // Show loader using BebanStoreLoader with custom color and animation type
        if (typeof BebanStoreLoader !== 'undefined') {
            const animationType = bebanWishlistAjax.animation_type || 'dots';
            const overlayColor = bebanWishlistAjax.remove_loader_overlay_color || 'rgba(255, 255, 255, 1)';
            const animationColor = bebanWishlistAjax.remove_loader_animation_color || '#81858B';
            
            if (animationType === 'none') {
                // بدون انیمیشن: فقط غیرفعال کردن دکمه
                $btn.addClass('beban-disabled');
            } else {
                // انیمیشن‌های بصری: spinner, dots, text
                const customText = animationType === 'text' ? bebanWishlistAjax.i18n.removing : null;
                BebanStoreLoader.showLoaderWithCustomColors($btn[0], animationType, animationColor, overlayColor, customText);
            }
        } else {
            $btn.addClass('beban-disabled');
        }
        var product_id = $btn.data('product-id');
        
        $.ajax({
            url: bebanWishlistAjax.ajax_url,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'beban_handle_wishlist',
                product_id: product_id,
                action_type: 'remove',
                nonce: bebanWishlistAjax.nonce
            },
                        success: function(response) {
                        if (response.success) {
                            // Hide loader first and restore button state
                            if (typeof BebanStoreLoader !== 'undefined') {
                                const animationType = bebanWishlistAjax.animation_type || 'dots';
                                
                                if (animationType === 'none') {
                                    // بدون انیمیشن: فقط فعال کردن دکمه
                                    $btn.removeClass('beban-disabled');
                                } else {
                                    // پنهان کردن loader برای انیمیشن‌های بصری
                                    BebanStoreLoader.hideLoader($btn[0]);
                                }
                            }
                            
                            // Update all wishlist buttons for this product across all pages
                            var $allProductButtons = $('.beban-wishlist-btn[data-product-id="' + product_id + '"]');
                            
                            // Remove the entire product card from the list with animation
                            $btn.closest('.beban-wishlist-item').fadeOut(300, function() {
                                $(this).remove();
                                
                                // Check if no products left
                                if ($('.beban-wishlist-item').length === 0) {
                                    $('.beban-wishlist-products').html(
                                        '<div style="text-align:center; margin-top:50px;">' +
                                        '<svg width="50" height="50" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="display:block; margin:0 auto 12px;">' +
                                        '<path d="M6.10999 17.5C3.89999 15.43 2 12.48 2 8.67999C2 5.58999 4.49 3.09003 7.56 3.09003C9.38 3.09003 10.99 3.97002 12 5.33002C13.01 3.97002 14.63 3.09003 16.44 3.09003C17.59 3.09003 18.66 3.43999 19.55 4.04999" stroke="rgba(41,45,50,0.3)" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>' +
                                        '<path d="M21.74 7C21.91 7.53 22 8.1 22 8.69C22 15.69 15.52 19.82 12.62 20.82C12.28 20.94 11.72 20.94 11.38 20.82C10.73 20.6 9.91002 20.22 9.02002 19.69" stroke="rgba(41,45,50,0.3)" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>' +
                                        '<path d="M22 2L2 22" stroke="#999" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>' +
                                        '</svg>' +
                                        '<p style="color:#999;">' + bebanWishlistAjax.empty_message + '</p>' +
                                        '</div>'
                                    );
                                }
                            });
                            
                            // Update all wishlist buttons to show empty heart
                            $allProductButtons.removeClass('beban-wishlisted');
                            $allProductButtons.find('svg').remove();
                            
                            // استفاده از رنگ‌های تنظیمات برای حالت عادی
                            const normalIcon = bebanWishlistAjax.wishlist_btn_normal_icon || '#A9ABAD';
                            const normalBg = bebanWishlistAjax.wishlist_btn_normal_bg || 'transparent';
                            
                            // ساخت SVG با circle به عنوان پس‌زمینه
                            let svgContent = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">';
                            if (normalBg !== 'transparent') {
                                svgContent += '<circle cx="12" cy="12" r="11" fill="none"/>';
                            }
                            svgContent += '<path d="M12.62 20.8101C12.28 20.9301 11.72 20.9301 11.38 20.8101C8.48 19.8201 2 15.6901 2 8.6901C2 5.6001 4.49 3.1001 7.56 3.1001C9.38 3.1001 10.99 3.9801 12 5.3401C13.01 3.9801 14.63 3.1001 16.44 3.1001C19.51 3.1001 22 5.6001 22 8.6901C22 15.6901 15.52 19.8201 12.62 20.8101Z" stroke="' + normalIcon + '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>';
                            
                            $allProductButtons.append(svgContent);
                            
                            // Remove loading class and re-enable buttons
                            $allProductButtons.removeClass('bebanstore-loading beban-disabled');
                            
                            // Re-enable all buttons after update
                            $allProductButtons.removeClass('beban-disabled');
                            
                            if (typeof BebanStoreToast !== 'undefined') {
                                if (response.data.status === 'already_removed') {
                                    BebanStoreToast.info(response.data.message);
                                } else {
                                    BebanStoreToast.warning(response.data.message);
                                }
                            }
                        } else {
                            if (typeof BebanStoreToast !== 'undefined') {
                                BebanStoreToast.error(response.data.message || 'خطایی رخ داد. لطفاً دوباره امتحان کنید.');
                            }
                        }
                    },
            error: function(xhr, status, error) {
                // Hide loader and restore button state
                if (typeof BebanStoreLoader !== 'undefined') {
                    const animationType = bebanWishlistAjax.animation_type || 'dots';
                    
                    if (animationType === 'none') {
                        // بدون انیمیشن: فقط فعال کردن دکمه
                        $btn.removeClass('beban-disabled');
                    } else {
                        // پنهان کردن loader برای انیمیشن‌های بصری
                        BebanStoreLoader.hideLoader($btn[0]);
                    }
                }
                
                if (typeof BebanStoreToast !== 'undefined') {
                    BebanStoreToast.error('خطایی رخ داد: ' + error);
                }
            },
            complete: function() {
                // Hide loader and restore button state
                if (typeof BebanStoreLoader !== 'undefined') {
                    BebanStoreLoader.hideLoader($btn[0]);
                } else {
                    $btn.removeClass('beban-disabled');
                }
            }
        });
    });
}); 