jQuery(function($) {
    'use strict';

    // Create modal at page load to ensure it's always available
    function createModal() {
        if ($('#bebanstore-remove-item-confirmation').length === 0) {
            // Get color settings
            var colors = typeof bebanstoreQuantity !== 'undefined' && bebanstoreQuantity.modalColors ? bebanstoreQuantity.modalColors : {
                modalBgColor: '#ffffff',
                confirmBgColor: '#dc3545',
                confirmTextColor: '#ffffff',
                cancelBgColor: '#ffffff',
                cancelTextColor: '#007bff',
                modalTitleColor: '#000000',
                modalDescriptionColor: '#666666'
            };
            
            var modalHtml = `
                <div id="bebanstore-remove-item-confirmation" class="bebanstore-modal">
                    <div class="bebanstore-modal-content" style="background-color: ${colors.modalBgColor} !important;">
                        <div class="bebanstore-modal-header">
                            <h3 style="color: ${colors.modalTitleColor} !important;">${typeof bebanstoreQuantity !== 'undefined' ? bebanstoreQuantity.i18n.confirmRemove : 'آیا مطمئن هستید؟'}</h3>
                        </div>
                        <div class="bebanstore-modal-body">
                            <p class="bebanstore-confirm-message" style="color: ${colors.modalDescriptionColor} !important;">${typeof bebanstoreQuantity !== 'undefined' ? bebanstoreQuantity.i18n.confirmRemoveMessage : 'آیا مطمئن هستید که می‌خواهید این محصول را از سبد خرید حذف کنید؟ این عملیات غیرقابل برگشت است.'}</p>
                        </div>
                        <div class="bebanstore-modal-footer">
                            <button class="bebanstore-modal-cancel" style="background-color: ${colors.cancelBgColor} !important; color: ${colors.cancelTextColor} !important;">${typeof bebanstoreQuantity !== 'undefined' ? bebanstoreQuantity.i18n.cancel : 'انصراف'}</button>
                            <button class="bebanstore-modal-confirm" style="background-color: ${colors.confirmBgColor} !important; color: ${colors.confirmTextColor} !important;">${typeof bebanstoreQuantity !== 'undefined' ? bebanstoreQuantity.i18n.remove : 'حذف'}</button>
                        </div>
                    </div>
                </div>
            `;
            $(modalHtml).appendTo('body');
        }
    }

    // Create modal immediately
    createModal();
    
    // Also create modal after a short delay to ensure it's available
    setTimeout(createModal, 100);

    // Handle quantity buttons
    function handleQuantityButtons() {
        // Find all quantity wrappers
        var $wrappers = $('.bebanstore-quantity-section');
        
        $wrappers.each(function(index) {
            var $wrapper = $(this);
            
            // Find elements within this specific wrapper
            var $quantity = $wrapper.find('.quantity');
            var $input = $wrapper.find('input.qty, input.bebanstore-qty-input');
            var $minusButton = $wrapper.find('.bebanstore-qty-button.minus');
            var $plusButton = $wrapper.find('.bebanstore-qty-button.plus');
            
            // Only proceed if we found all necessary elements
            if (!$input.length || !$minusButton.length || !$plusButton.length) {
                return;
            }
            
            var min = parseInt($input.attr('min') || 1);
            var max = parseInt($input.attr('max') || 0);
            var step = parseInt($input.attr('step') || 1);
            
            // Fix min value if it's 0 or negative
            if (min <= 0) {
                min = 1;
            }
            
            // Fix max value if it's negative
            if (max < 0) {
                max = 0; // 0 means no limit
            }

            // Remove any existing event handlers
            $minusButton.off('click');
            $plusButton.off('click');
            $input.off('change');

            // Handle minus button click
            $minusButton.on('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var currentVal = parseInt($input.val());
                var isInCart = $wrapper.closest('.woocommerce-cart-form').length > 0;
                
                // Check if trash button is enabled
                var enableTrashButton = typeof bebanstoreQuantity !== 'undefined' ? bebanstoreQuantity.enableTrashButton : false;
                
                // If in cart and quantity is 1 and trash button is enabled, remove the item
                if (isInCart && currentVal === 1 && enableTrashButton) {
                    // Find the cart item key
                    var cartItemKey = '';
                    var $cartItem = $wrapper.closest('.cart_item, .woocommerce-cart-form__cart-item');
                    
                    if ($cartItem.length) {
                        // Try to get cart item key from input name
                        var inputName = $input.attr('name');
                        if (inputName) {
                            var matches = inputName.match(/cart\[([^\]]+)\]\[qty\]/);
                            if (matches) {
                                cartItemKey = matches[1];
                            }
                        }
                        
                        // If not found, try to get from remove link
                        if (!cartItemKey) {
                            var $removeLink = $cartItem.find('.product-remove a, .remove');
                            if ($removeLink.length) {
                                var href = $removeLink.attr('href');
                                if (href) {
                                    var hrefMatches = href.match(/remove_item=([^&]+)/);
                                    if (hrefMatches) {
                                        cartItemKey = decodeURIComponent(hrefMatches[1]);
                                    }
                                }
                            }
                        }
                        
                        if (cartItemKey) {
                            // Check if confirmation is enabled
                            var enableConfirmation = typeof bebanstoreQuantity !== 'undefined' ? bebanstoreQuantity.enableRemoveConfirmation : true;
                            
                            if (enableConfirmation) {
                                // Show confirmation modal
                                showRemoveConfirmationModal(cartItemKey, $cartItem);
                            } else {
                                // Direct removal without confirmation
                                removeCartItem(cartItemKey, $cartItem);
                            }
                        }
                    }
                } else if (currentVal > min) {
                    // Normal quantity decrease
                    var newVal = currentVal - step;
                    
                    $input
                        .val(newVal)
                        .trigger('change')
                        .trigger('input')
                        .trigger('woocommerce_quantity_input_changed');
                        
                    updateButtonStates();
                    
                    // If in cart, trigger update
                    if (isInCart) {
                        scheduleCartUpdate();
                    }
                }
            });

            // Handle plus button click
            $plusButton.on('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var currentVal = parseInt($input.val());
                var max = parseInt($input.attr('max') || 0);
                
                // Check if we're already at max
                if (max > 0 && currentVal === (max - 1)) {
                    // Show warning toast when already at maximum quantity
                    if (typeof BebanStoreToast !== 'undefined') {
                        BebanStoreToast.info('حداکثر تعداد مجاز این محصول ' + max + ' عدد است');
                    }
                }
                
                // If not at max, increase the value
                if (max <= 0 || currentVal < max) {
                    var newVal = currentVal + step;
                    
                    $input
                        .val(newVal)
                        .trigger('change')
                        .trigger('input')
                        .trigger('woocommerce_quantity_input_changed');
                        
                    updateButtonStates();
                    
                    // If in cart, trigger update
                    if ($wrapper.closest('.woocommerce-cart-form').length) {
                        scheduleCartUpdate();
                    }
                }
            });

            // Update button states based on current value
            function updateButtonStates() {
                var currentVal = parseInt($input.val());
                var max = parseInt($input.attr('max') || 0);
                
                // Check if we're in cart context
                var isInCart = $wrapper.closest('.woocommerce-cart-form').length > 0;
                
                // Check if trash button is enabled
                var enableTrashButton = typeof bebanstoreQuantity !== 'undefined' ? bebanstoreQuantity.enableTrashButton : false;
                
                if (isInCart && currentVal === 1 && enableTrashButton) {
                    // Convert minus button to trash button
                    var trashIcon = $minusButton.data('trash-icon');
                    if (trashIcon) {
                        $minusButton.html(trashIcon);
                        $minusButton.attr('aria-label', 'حذف محصول');
                        $minusButton.removeClass('minus').addClass('trash');
                    }
                } else {
                    // Convert back to minus button
                    var minusIcon = $minusButton.data('minus-icon');
                    if (minusIcon) {
                        $minusButton.html(minusIcon);
                        $minusButton.attr('aria-label', 'کاهش تعداد');
                        $minusButton.removeClass('trash').addClass('minus');
                    }
                }
                
                // Disable minus button if at min (but not in cart with quantity 1)
                var minusDisabled = currentVal <= min && !(isInCart && currentVal === 1 && enableTrashButton);
                $minusButton.prop('disabled', minusDisabled);
                
                // Disable plus button if at max (only if max is set and greater than 0)
                var plusDisabled = max > 0 && currentVal >= max;
                $plusButton.prop('disabled', plusDisabled);
            }

            // Initial button state
            updateButtonStates();
        });
    }

    // Schedule cart update with debounce
    var updateTimer;
    function scheduleCartUpdate() {
        clearTimeout(updateTimer);
        updateTimer = setTimeout(function() {
            $('[name="update_cart"]').prop('disabled', false).trigger('click');
        }, 500);
    }

    // Function to remove cart item
    function removeCartItem(cartItemKey, $cartItem) {
        $.ajax({
            url: bebanstoreQuantity.ajaxurl,
            type: 'POST',
            data: {
                action: 'bebanstore_remove_cart_item',
                cart_item_key: cartItemKey,
                nonce: bebanstoreQuantity.nonce
            },
            success: function(response) {
                if (response.success) {
                    console.log('BebanStore: Remove cart item response', response);
                    
                    // Remove item from DOM with animation
                    $cartItem.fadeOut(300, function() {
                        $(this).remove();
                        
                        // Check if cart is empty
                        var remainingItems = $('.cart_item, .woocommerce-cart-form__cart-item').length;
                        if (remainingItems === 0) {
                            $('.woocommerce-cart-form').replaceWith('<div class="woocommerce"><div class="woocommerce-notices-wrapper"></div><p class="cart-empty woocommerce-info">سبد خرید شما خالی است.</p><p class="return-to-shop"><a class="button wc-backward" href="' + wc_cart_fragments_params.wc_ajax_url.replace('%%endpoint%%', 'get_refreshed_fragments') + '">بازگشت به فروشگاه</a></p></div>');
                        } else {
                            // Use fragments from response if available
                            if (response.data && response.data.fragments) {
                                console.log('BebanStore: Updating fragments from response', response.data.fragments);
                                
                                $.each(response.data.fragments, function(key, value) {
                                    console.log('BebanStore: Replacing', key);
                                    $(key).replaceWith(value);
                                });
                                
                                // Trigger WooCommerce events
                                $(document.body).trigger('wc_fragments_refreshed');
                                $(document.body).trigger('updated_cart_totals');
                                $(document.body).trigger('wc_fragment_refresh');
                                
                                if (response.data.cart_hash) {
                                    $(document.body).trigger('wc_cart_hash_changed', [response.data.cart_hash]);
                                }
                            } else {
                                console.log('BebanStore: No fragments in response, requesting refresh');
                                
                                // Fallback: Refresh cart fragments to update totals
                                if (typeof wc_cart_fragments_params !== 'undefined') {
                                    $.ajax({
                                        url: wc_cart_fragments_params.wc_ajax_url.toString().replace('%%endpoint%%', 'get_refreshed_fragments'),
                                        type: 'POST',
                                        success: function(data) {
                                            if (data && data.fragments) {
                                                console.log('BebanStore: Fallback fragments received', data.fragments);
                                                
                                                $.each(data.fragments, function(key, value) {
                                                    $(key).replaceWith(value);
                                                });
                                                
                                                // Trigger WooCommerce events
                                                $(document.body).trigger('wc_fragments_refreshed');
                                                $(document.body).trigger('updated_cart_totals');
                                                $(document.body).trigger('wc_fragment_refresh');
                                                
                                                if (data.cart_hash) {
                                                    $(document.body).trigger('wc_cart_hash_changed', [data.cart_hash]);
                                                }
                                            }
                                        }
                                    });
                                }
                            }
                        }
                    });
                    
                    // Show success toast
                    if (typeof BebanStoreToast !== 'undefined') {
                        BebanStoreToast.warning('محصول از سبد خرید حذف شد');
                    }
                } else {
                    console.error('BebanStore: Remove cart item failed', response);
                    if (typeof BebanStoreToast !== 'undefined') {
                        BebanStoreToast.error('خطا در حذف محصول');
                    }
                }
            },
            error: function() {
                if (typeof BebanStoreToast !== 'undefined') {
                    BebanStoreToast.error('خطا در حذف محصول');
                }
            }
        });
    }

    // Function to show confirmation modal for removing cart item
    function showRemoveConfirmationModal(cartItemKey, $cartItem) {
        var $modal = $('#bebanstore-remove-item-confirmation');
        
        // Ensure modal exists
        if ($modal.length === 0) {
            createModal();
            $modal = $('#bebanstore-remove-item-confirmation');
        }
        
        // Store data in modal
        $modal.data('cart-item-key', cartItemKey);
        $modal.data('cart-item-element', $cartItem);
        
        // Show modal
        $modal.addClass('show');
    }

    // Initialize on page load
    handleQuantityButtons();

    // Initialize when cart is updated
    $(document.body).on('updated_cart_totals', function() {
        handleQuantityButtons();
    });

    // Initialize when variations are loaded
    $(document.body).on('found_variation', function(event, variation) {
        handleQuantityButtons();
        
        // Update max value based on variation stock
        if (variation.max_qty) {
            $('.qty, .bebanstore-qty-input').attr('max', variation.max_qty);
        }
    });

    // Initialize when cart fragments are updated
    $(document.body).on('wc_fragments_refreshed', function() {
        handleQuantityButtons();
    });

    // Initialize when checkout is updated
    $(document.body).on('updated_checkout', function() {
        handleQuantityButtons();
    });

    // Handle modal events using event delegation
    $(document).on('click', '#bebanstore-remove-item-confirmation .bebanstore-modal-cancel', function() {
        $('#bebanstore-remove-item-confirmation').removeClass('show');
    });

    $(document).on('click', '#bebanstore-remove-item-confirmation .bebanstore-modal-confirm', function() {
        var $modal = $('#bebanstore-remove-item-confirmation');
        var cartItemKey = $modal.data('cart-item-key');
        var $cartItem = $modal.data('cart-item-element');
        
        if (cartItemKey && $cartItem) {
            removeCartItem(cartItemKey, $cartItem);
        }
        
        // Hide modal
        $modal.removeClass('show');
    });

    // Close modal when clicking outside
    $(document).on('click', '#bebanstore-remove-item-confirmation', function(e) {
        if (e.target === this) {
            $(this).removeClass('show');
        }
    });

    // Close modal on escape key
    $(document).on('keyup', function(e) {
        if (e.key === 'Escape' && $('#bebanstore-remove-item-confirmation').hasClass('show')) {
            $('#bebanstore-remove-item-confirmation').removeClass('show');
        }
    });
}); 