/**
 * BebanStore Loader Utility
 * Manages loading states for all BebanStore features
 */
(function($) {
    'use strict';

    const BebanStoreLoader = {
        /**
         * Configuration
         */
        config: {
            duration: 300,
            loaderSize: 24,
            loaderColor: '#ef394e',
            // Loader types: 'dots', 'spinner', 'pulse'
            defaultLoaderType: 'dots',

        },

        /**
         * Current loader type
         */
        currentLoaderType: 'dots',

        /**
         * Set loader type
         * @param {string} type - Loader type ('dots', 'spinner', 'pulse')
         */
        setLoaderType: function(type) {
            if (['dots', 'spinner', 'pulse'].includes(type)) {
                this.currentLoaderType = type;
            }
        },

        /**
         * Get current loader type
         * @returns {string} - Current loader type
         */
        getLoaderType: function() {
            return this.currentLoaderType;
        },



        /**
         * Show loader on target element
         * @param {HTMLElement} targetElement - Element to show loader on
         * @param {string} type - Type of loader ('default', 'dots', 'spinner', 'pulse')
         * @param {string} color - Custom color for loader (optional)
         */
        showLoader: function(targetElement, type = 'default', color = null) {
            if (!targetElement) return;

            const $element = $(targetElement);
            
            // Add loading class
            $element.addClass('bebanstore-loading');
            
            // Store original position and ensure relative positioning for overlay
            const originalPosition = $element.css('position');
            if (originalPosition === 'static') {
                $element.data('original-position', originalPosition);
                $element.css('position', 'relative');
            }

            // Determine loader type
            const loaderType = (type === 'default') ? this.currentLoaderType : type;
            
            // Create loader HTML based on type
            const loaderHTML = this.createLoaderHTML(loaderType, color);
            
            // Create overlay loader
            const $overlay = $(`<div class="bebanstore-loader-overlay">${loaderHTML}</div>`);
            
            // Add overlay to element
            $element.append($overlay);
            
            // Disable element if it's a button
            if ($element.is('button')) {
                $element.prop('disabled', true);
            }
        },

        /**
         * Show loader with custom colors
         * @param {HTMLElement} targetElement - Element to show loader on
         * @param {string} type - Type of loader ('default', 'dots', 'spinner', 'pulse', 'text')
         * @param {string} animationColor - Color for animation
         * @param {string} overlayColor - Color for overlay background
         * @param {string} customText - Custom text for text loader
         */
        showLoaderWithCustomColors: function(targetElement, type = 'default', animationColor = null, overlayColor = null, customText = null) {
            if (!targetElement) return;

            const $element = $(targetElement);
            
            // Add loading class
            $element.addClass('bebanstore-loading');
            
            // Store original position and ensure relative positioning for overlay
            const originalPosition = $element.css('position');
            if (originalPosition === 'static') {
                $element.data('original-position', originalPosition);
                $element.css('position', 'relative');
            }

            // Determine loader type
            const loaderType = (type === 'default') ? this.currentLoaderType : type;
            
            // Create loader HTML based on type
            const loaderHTML = this.createLoaderHTML(loaderType, animationColor, customText);
            
            // Create overlay loader with custom background color
            const $overlay = $(`<div class="bebanstore-loader-overlay">${loaderHTML}</div>`);
            
            // Apply custom overlay color if provided
            if (overlayColor) {
                $overlay.css('background-color', overlayColor);
            }
            
            // Add overlay to element
            $element.append($overlay);
            
            // Disable element if it's a button
            if ($element.is('button')) {
                $element.prop('disabled', true);
            }
        },



        /**
         * Execute AJAX request with automatic loader management
         * @param {Object} options - AJAX options
         * @param {HTMLElement} loaderTarget - Element to show loader on
         * @param {Function} onSuccess - Success callback
         * @param {Function} onError - Error callback
         */
        executeWithLoader: function(options, loaderTarget, onSuccess, onError) {
            if (!loaderTarget) {
                // Execute without loader if no target provided
                $.ajax(options);
                return;
            }

            // Show loader
            this.showLoader(loaderTarget);

            // Store original callbacks
            const originalSuccess = options.success;
            const originalError = options.error;
            const originalComplete = options.complete;

            // Override callbacks to handle loader
            options.success = function(response) {
                // Hide loader
                BebanStoreLoader.hideLoader(loaderTarget);
                
                // Call original success
                if (originalSuccess) {
                    originalSuccess(response);
                }
                
                // Call custom success
                if (onSuccess) {
                    onSuccess(response);
                }
            };

            options.error = function(xhr, status, error) {
                // Hide loader
                BebanStoreLoader.hideLoader(loaderTarget);
                
                // Call original error
                if (originalError) {
                    originalError(xhr, status, error);
                }
                
                // Call custom error
                if (onError) {
                    onError(xhr, status, error);
                }
            };

            options.complete = function() {
                // Call original complete
                if (originalComplete) {
                    originalComplete();
                }
            };

            // Execute AJAX
            $.ajax(options);
        },

        /**
         * Hide loader and restore original content
         * @param {HTMLElement} targetElement - Element to hide loader from
         */
        hideLoader: function(targetElement) {
            if (!targetElement) return;

            const $element = $(targetElement);
            
            // Remove loading class
            $element.removeClass('bebanstore-loading');
            
            // Remove overlay loader
            $element.find('.bebanstore-loader-overlay').remove();
            
            // Reset position if it was changed
            if ($element.data('original-position')) {
                $element.css('position', $element.data('original-position'));
                $element.removeData('original-position');
            }
            
            // Re-enable element if it's a button
            if ($element.is('button')) {
                $element.prop('disabled', false);
            }
        },

        /**
         * Create loader HTML based on type
         * @param {string} type - Type of loader
         * @param {string} color - Custom color for loader
         * @returns {string} - HTML string for loader
         */
        createLoaderHTML: function(type, color = null, text = null) {
            const colorStyle = color ? `style="color: ${color};"` : '';
            
            switch (type) {
                case 'dots':
                    return `<div class="bebanstore-dots" ${colorStyle}>
                        <div class="dot"></div>
                        <div class="dot"></div>
                        <div class="dot"></div>
                    </div>`;
                
                case 'spinner':
                    return `<div class="bebanstore-spinner" ${colorStyle}></div>`;
                
                case 'pulse':
                    return `<div class="bebanstore-pulse" ${colorStyle}></div>`;
                
                case 'text':
                    const loadingText = text || 'در حال بارگذاری...';
                    return `<div class="bebanstore-text-loader" ${colorStyle}>${loadingText}</div>`;
                
                default:
                    // Fallback to dots if type is not recognized
                    return `<div class="bebanstore-dots" ${colorStyle}>
                        <div class="dot"></div>
                        <div class="dot"></div>
                        <div class="dot"></div>
                    </div>`;
            }
        },



        /**
         * Initialize loader system
         */
        init: function() {
            // Set initial loader type from config
            this.currentLoaderType = this.config.defaultLoaderType;
        }
    };

    // Export to global scope
    window.BebanStoreLoader = BebanStoreLoader;

    // Initialize on document ready
    $(document).ready(function() {
        BebanStoreLoader.init();
    });

})(jQuery); 