<?php
/**
 * Feature Access Manager Class
 * 
 * Manages access levels and permissions for BebanStore features,
 * including Free, Pro, and Mixed feature types.
 * 
 * @package BebanStore
 * @since 1.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class BebanStore_Feature_Access_Manager {
    /**
     * Pro features registry
     * 
     * @var array
     * @access private
     */
    private $pro_features = array();

    /**
     * Mixed features registry
     * 
     * @var array
     * @access private
     */
    private $mixed_features = array();

    /**
     * Initialize the class
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Allow Pro features to register themselves
        add_filter('bebanstore_pro_feature_classes', array($this, 'register_pro_features'));
        
        // Hook for Pro availability check
        add_filter('bebanstore_pro_available', array($this, 'is_pro_activated'));
    }

    /**
     * Register Pro features
     * 
     * @param array $features Existing Pro features
     * @return array Updated Pro features
     */
    public function register_pro_features($features) {
        return array_merge($features, $this->pro_features);
    }

    /**
     * Check if a feature is Pro-only
     * 
     * @param string $feature_id Feature ID
     * @return bool Whether the feature is Pro-only
     */
    public function is_pro_feature($feature_id) {
        return isset($this->pro_features[$feature_id]);
    }

    /**
     * Check if a feature is mixed (has both free and Pro settings)
     * 
     * @param string $feature_id Feature ID
     * @return bool Whether the feature is mixed
     */
    public function is_mixed_feature($feature_id) {
        return in_array($feature_id, $this->mixed_features);
    }

    /**
     * Check if Pro is activated
     * 
     * @return bool Whether Pro is activated
     */
    public function is_pro_activated() {
        // Check if Pro plugin class exists and is activated
        if (class_exists('BebanStore_Pro')) {
            return BebanStore_Pro::is_activated();
        }
        
        // Check via filter for custom activation logic
        return apply_filters('bebanstore_pro_activation_check', false);
    }

    /**
     * Get Pro features count
     * 
     * @return int Number of Pro features
     */
    public function get_pro_features_count() {
        return count($this->pro_features);
    }

    /**
     * Register a Pro feature
     * 
     * @param string $feature_id Feature ID
     * @param string $class_name Feature class name
     */
    public function register_pro_feature($feature_id, $class_name) {
        $this->pro_features[$feature_id] = $class_name;
    }

    /**
     * Register a mixed feature
     * 
     * @param string $feature_id Feature ID
     */
    public function register_mixed_feature($feature_id) {
        if (!in_array($feature_id, $this->mixed_features)) {
            $this->mixed_features[] = $feature_id;
        }
    }
}
